#!/usr/bin/env python

from __future__ import unicode_literals, absolute_import, print_function

import ssl
import json
import argparse
import uuid
import requests
from requests.adapters import HTTPAdapter


def parse_args():
    parser = argparse.ArgumentParser()
    igroup = parser.add_mutually_exclusive_group(required=True)
    igroup.add_argument("-f", "--file", help='Input file')
    igroup.add_argument("-t", "--targets", help='Targets')
    parser.add_argument("-o", "--output", help='Output file name')
    return parser.parse_args()


def read_targets_from_file(filepath):
    """Read targets from file. File must have a specific format: json dict per line:
    {"result":{"dest_ip":"127.0.0.1", "dest_port":[443]}}
    {"result":{"dest_ip":"127.0.0.2", "dest_port":[443]}}
    """
    fp = open(filepath, 'r')
    data = fp.read()
    fp.close()

    results = list()

    for line in data.split('\n'):
        try:
            line = json.loads(line)
        except ValueError:
            continue

        result = line.get('result')
        if not result:
            continue

        ip = result.get('dest_ip')
        ports = result.get('ports')
        if not ip or not ports:
            continue

        if not isinstance(ports, list):
            ports = [ports]

        for port in ports:
            if port.isdigit():
                results.append((ip, int(port)))

    return results


def read_targets_from_cmdline(targetsline):
    results = list()
    targets = targetsline.split(',')

    for target in targets:
        stripped = target.strip()
        if stripped[0] == '[' and stripped[-1] == ']':
            results.append((stripped, 2375))
            continue

        if stripped.count(":") >= 2:
            ip = "[" + stripped + "]"
            results.append((ip, 2375))
            continue

        # splitted = stripped.rsplit(':', 1)
        # if len(splitted) == 2 and splitted[1].isdigit():
        #     results.append((splitted[0], int(splitted[1])))
        # else:
        results.append((stripped, 2375))

    return results


def try_docker(ip, port):
    result = {
        'ip': ip,
        'port': port,
        'is_docker': None,
        'unknown_exception': None,
    }

    if ip and ip[0] == '[' and ip[-1] == ']':
        result["ip"] = ip[1:-1]

    session = requests.Session()
    session.mount('http://', HTTPAdapter(max_retries=3))

    query_version = "http://{host}:{port}/v1.18/version".format(host=ip, port=port)

    try:
        resp = session.get(query_version, timeout=30)
        if b'docker' in resp.content:
            result["is_docker"] = True
        elif b'MinAPIVersion' in resp.content:
            result["is_docker"] = True
        else:
            result["is_docker"] = False

        result['unknown_exception'] = False
    except KeyboardInterrupt as e:
        raise e

    except Exception as e:
        # import traceback; traceback.print_exc()
        result['unknown_exception'] = "{}".format(e)

    # print(result)
    return result


def check_one(ip, port):
    res = try_docker(ip, port)
    
    return res


def check_all(targets):
    results = list()

    for target in targets:
        res = check_one(target[0], target[1])
        results.append(res)

    return results


def main():
    args = parse_args()

    if args.file:
        targets = read_targets_from_file(args.file)
    else:
        targets = read_targets_from_cmdline(args.targets)

    results = check_all(targets)

    if not args.output:
        print(json.dumps(results))
    else:
        fp = open(args.output, 'w')
        fp.write(json.dumps(results))
        fp.close()


if __name__ == '__main__':
    main()
