#!/usr/bin/env python

from __future__ import unicode_literals, absolute_import, print_function

import json
import argparse
import subprocess
from threading import Timer


IPMI_DEFAULT_PORT = 623
DEFAULTS = ["ADMIN", "admin"]


def parse_args():
    """
    Parse script input args.
    Accepts target list and path to output file.
    """
    parser = argparse.ArgumentParser()
    parser.add_argument("-t", "--targets", 
                        help='list of target-port pairs: "localhost:8123,127.0.0.1:8122,localhost"')
    parser.add_argument("-o", "--output", help='Output file name')
    return parser.parse_args()


def try_ipmi_imp(ip, port, username, passwd):
    cmd = [
        "ipmitool", "-I", "lanplus", "-H", ip, "-p", str(port), "-C", "0",
        "-U", username, "-P", passwd, "lan", "print"
    ]
    process = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    kill = lambda process: process.kill()

    # Execution timeout
    stdout = ''
    stderr = ''
    my_timer = Timer(5, kill, [process])
    try:
        my_timer.start()
        stdout, stderr = process.communicate()
    finally:
        my_timer.cancel()
    
    # check Error: ... in stderr and something in stdout
    if not stdout or "Error" in stderr:
        return False
    else:
        return True


def try_ipmi(ip, port):

    for username in DEFAULTS:
        for passwd in DEFAULTS:
            found = try_ipmi_imp(ip, port, username, passwd)
            if found:
                return True

    return False


def check_all(targets):
    results = list()
    for target in targets:

        target_info = {
            'target': target[0],
            'port': target[1],
            'enabled': try_ipmi(target[0], target[1]),
        }

        results.append(target_info)

    return results


def prepare_targets(targets):
    """
    Split targets string to list of (host, port) pairs.

    Example:
    ```
    >>> prepare_targets("127.0.0.1:623,127.0.0.2")
    [("127.0.0.1", 623),("127.0.0.2", 623)]
    ```
    """
    results = list()

    elems = targets.split(',')
    for elem in elems:
        stripped = elem.strip()
        parts = stripped.rsplit(':', 1)
        host = parts[0]
        port = int(parts[-1]) if len(parts) == 2 else IPMI_DEFAULT_PORT
        results.append((host, port))

    return results


def main():
    args = parse_args()

    targets = prepare_targets(args.targets)
    results = check_all(targets)

    if not args.output:
        print(json.dumps(results))
    else:
        fp = open(args.output, 'w')
        fp.write(json.dumps(results))
        fp.close()


if __name__ == '__main__':
    main()
