#!/usr/bin/env python

from __future__ import unicode_literals, absolute_import, print_function

import ssl
import json
import argparse
import ipaddress
from pymongo import MongoClient
from pymongo.errors import ConfigurationError, ServerSelectionTimeoutError
from pymongo.errors import AutoReconnect, OperationFailure


def parse_args():
    parser = argparse.ArgumentParser()
    igroup = parser.add_mutually_exclusive_group(required=True)
    igroup.add_argument("-f", "--file", help='Input file')
    igroup.add_argument("-t", "--targets", help='Targets')
    parser.add_argument("-o", "--output", help='Output file name')
    return parser.parse_args()


def prepare_host(addr, port=27017):
    decoded_addr = addr.decode()
    prepare_addr = addr
    
    try:
        if ipaddress.ip_address(decoded_addr).version == 6:
            prepare_addr = '[{}]'.format(addr)
    except ValueError:
        pass

    if port != 27017:
        port_ = ":{}".format(port)
    else:
        port_ = ''

    return '{}{}'.format(prepare_addr, port_)


def read_targets_from_file(filepath):
    fp = open(filepath, 'r')
    data = fp.read()
    fp.close()

    results = list()

    for line in data.split('\n'):
        try:
            line = json.loads(line)
        except ValueError:
            continue

        result = line.get('result')
        if not result:
            continue

        ip = result.get('dest_ip')
        ports = result.get('ports')
        if not ip or not ports:
            continue

        if not isinstance(ports, list):
            ports = [ports]

        for port in ports:
            if port.isdigit():
                results.append((ip, int(port)))

    return results


def read_targets_from_cmdline(targetsline):
    results = list()
    targets = targetsline.split(',')

    for target in targets:
        stripped = target.strip()
        splitted = stripped.rsplit(':', 1)
        if len(splitted) == 2 and splitted[1].isdigit():
            results.append((splitted[0], int(splitted[1])))
        else:
            results.append((stripped, 27017))
            results.append((stripped, 27018))

    return results


def try_mongo(ip, port, use_ssl=False):

    host = prepare_host(ip, port)
    url = "mongodb://{}".format(host)

    result = {
        'ip': ip[1:-1] if ip[0] == '[' and ip[-1] == ']' else ip,
        'port': port,
        'enabled': True,
        'version': None,
        'auth_required': None,
        'unknown_exception': False,
    }

    if use_ssl:
        client = MongoClient(url, serverSelectionTimeoutMS=2000, ssl=True, ssl_cert_reqs=ssl.CERT_NONE)
    else:
        client = MongoClient(url, serverSelectionTimeoutMS=3000)

    try:
        r = client.server_info()
        ver = r.get('version')
        if ver:
            result['version'] = ver
            result['enabled'] = True

        r = client.database_names()
        result['auth_required'] = False
        result['enabled'] = True

    except OperationFailure as e:
        result['enabled'] = True

        if "not authorized" in e.message:
            result['auth_required'] = True
        else:
            result['unknown_exception'] = True

    except ConfigurationError as e:
        result['enabled'] = False
        pass

    except ServerSelectionTimeoutError as e:
        result['enabled'] = False
        pass

    except KeyError as e:
        result['enabled'] = False
        pass

    except AutoReconnect as e:
        result['enabled'] = False
        pass

    return result


def check_one(ip, port):
    res = try_mongo(ip, port, use_ssl=False)
    if res['enabled']:
        res['ssl'] = False
        return res

    res = try_mongo(ip, port, use_ssl=True)
    if res['enabled']:
        res['ssl'] = True
        return res

    res['ssl'] = None
    
    return res

def check_all(targets):
    results = list()

    for target in targets:
        res = check_one(target[0], target[1])
        # print(res)
        results.append(res)

    return results


def main():
    args = parse_args()

    if args.file:
        targets = read_targets_from_file(args.file)
    else:
        targets = read_targets_from_cmdline(args.targets)

    results = check_all(targets)

    if not args.output:
        print(json.dumps(results))
    else:
        fp = open(args.output, 'w')
        fp.write(json.dumps(results))
        fp.close()


if __name__ == '__main__':
    main()
