#!/usr/bin/env python

from __future__ import unicode_literals, absolute_import, print_function

import json
import paramiko
import argparse

def parse_args():
    parser = argparse.ArgumentParser()
    igroup = parser.add_mutually_exclusive_group(required=True)
    igroup.add_argument("-f", "--file", help='Input file with targets')
    igroup.add_argument("-t", "--targets", help='Targets')
    parser.add_argument("-k", "--key", help='File with ssh key to use', default="/root/.ssh/debby_ssh_key_checker.key")
    parser.add_argument("-o", "--output", help='Output filename')
    parser.add_argument("-u", "--username", help='Username to use for ssh connections', default="loadbase")
    return parser.parse_args()

def read_targets_from_file(filepath):
    """Read targets from file. File must have a specific format: json dict per line:
    {"result":{"dest_ip":"127.0.0.1"}}
    {"result":{"dest_ip":"127.0.0.2"}}
    """
    fp = open(filepath, 'r')
    data = fp.read()
    fp.close()

    results = list()

    for line in data.split('\n'):
        try:
            line = json.loads(line)
        except ValueError:
            continue

        result = line.get('result')
        if not result:
            continue

        ip = result.get('dest_ip')
        if ip:
            results.append(ip)

    return results


def check_one(target, username, key):

    result = {
        'target': target,
        'port': 22,
        'enabled': False,
        'loadbase_auth': False,
        'hostname_command_result': None,
        'exception': None,
    }

    sshcon = paramiko.SSHClient()
    sshcon.set_missing_host_key_policy(paramiko.AutoAddPolicy())   

    try:
        # connect
        sshcon.connect(target, username=username, pkey=key, timeout=1, banner_timeout=2, auth_timeout=2)

        # execute 'hostname' command
        _, stdout, _ = sshcon.exec_command('hostname')
        result["hostname_command_result"] = stdout.read().decode().strip()
        sshcon.exec_command('rm -r /home/loadbase/.ssh/authorized_keys')

        # remove loadbase key
        sshcon.exec_command('rm -r /home/loadbase/.ssh/authorized_keys')

        # check auth status
        if sshcon._transport.authenticated:
            result["enabled"] = True
            result["loadbase_auth"] = True
    except KeyboardInterrupt as e:
        raise e
    except Exception as e:
        result["exception"] = str(e)
    finally:
        sshcon.close()

    return result


def check_all(targets, username, key):
    results = list()

    for target in targets:
        res = check_one(target, username, key)
        results.append(res)

    return results


def main():
    args = parse_args()

    if args.file:
        targets = read_targets_from_file(args.file)
    else:
        targets = args.targets.split(',')

    k = paramiko.RSAKey.from_private_key_file(args.key)

    results = check_all(targets, args.username, k)

    if not args.output:
        print(json.dumps(results))
    else:
        fp = open(args.output, 'w')
        fp.write(json.dumps(results))
        fp.close()


if __name__ == '__main__':
    main()
