#!/usr/bin/env python

from __future__ import unicode_literals, absolute_import, print_function

import json
import argparse
import kazoo
from kazoo.client import KazooClient


def parse_args():
    parser = argparse.ArgumentParser()
    igroup = parser.add_mutually_exclusive_group(required=True)
    igroup.add_argument("-f", "--file", help='Input file')
    igroup.add_argument("-t", "--targets", help='Targets')
    parser.add_argument("-o", "--output", help='Output file name')
    return parser.parse_args()


def read_targets_from_file(filepath):
    """Read targets from file. File must have a specific format: json dict
    per line:
    {"result":{"dest_ip":"127.0.0.1", "dest_port":[443]}}
    {"result":{"dest_ip":"127.0.0.2", "dest_port":[443]}}
    """
    fp = open(filepath, 'r')
    data = fp.read()
    fp.close()

    results = list()

    for line in data.split('\n'):
        try:
            line = json.loads(line)
        except ValueError:
            continue

        result = line.get('result')
        if not result:
            continue

        ip = result.get('dest_ip')
        ports = result.get('ports')
        if not ip or not ports:
            continue

        if not isinstance(ports, list):
            ports = [ports]

        for port in ports:
            if port.isdigit():
                results.append((ip, int(port)))

    return results


def read_targets_from_cmdline(targetsline):
    results = list()
    targets = targetsline.split(',')

    for target in targets:
        stripped = target.strip()
        if stripped[0] == '[' and stripped[-1] == ']':
            results.append((stripped, 2181))
            continue

        if stripped.count(":") >= 2:
            ip = "[" + stripped + "]"
            results.append((ip, 2181))
            continue

        results.append((stripped, 2181))
    return results


def try_docker(ip, port):
    result = {
        'ip': ip,
        'port': port,
        'auth_required': None,
        'exception': None,
        'command_result': None,
        'unknown_exception': None,
        'enabled': True,
    }

    if ip and ip[0] == '[' and ip[-1] == ']':
        result["ip"] = ip[1:-1]

    host = '{host}:{port}'.format(host=ip, port=port)
    zk = KazooClient(hosts=host, read_only=True,  verify_certs=False, timeout=5)
    try:
        zk.start()
        command_result = zk.get_children('/')
        result['auth_required'] = False
        result['command_result'] = command_result

    except kazoo.handlers.threading.KazooTimeoutError as e:
        result['exception'] = "kazoo.handlers.threading.KazooTimeoutError: {}".format(e)
        result['enabled'] = False

    except kazoo.exceptions.NoAuthError as e:
        result['exception'] = "kazoo.exceptions.NoAuthError: {}".format(e)
        result['auth_required'] = True

    except Exception as e:
        result['unknown_exception'] = "Exception: {}".format(e)
        result['enabled'] = False
    
    zk.stop()

    return result


def check_one(ip, port):
    res = try_docker(ip, port)
    return res


def check_all(targets):
    results = list()

    for target in targets:
        res = check_one(target[0], target[1])
        results.append(res)
    return results


def main():
    args = parse_args()

    if args.file:
        targets = read_targets_from_file(args.file)
    else:
        targets = read_targets_from_cmdline(args.targets)

    results = check_all(targets)

    if not args.output:
        print(json.dumps(results))
    else:
        fp = open(args.output, 'w')
        fp.write(json.dumps(results))
        fp.close()


if __name__ == '__main__':
    main()

