

import traceback
from flask import Blueprint, redirect, url_for, render_template, request, abort, jsonify

from app.db.models import DebbyAgent, DebbyTag, RelationAgentTag
from app.db.db import new_session, lock_safe_query
from app import yauth
from app.validators import DebbyValidateException
from app.agents.validators import validate_agent_data


bp = Blueprint('agents', __name__, url_prefix='/agents')


@bp.route('/', methods=['GET'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def agents_main():
    session = new_session()
    agents = session.query(DebbyAgent).all()
    tags = session.query(DebbyTag).all()
    for agent in agents:
        agent_tags = session.query(DebbyTag).filter(DebbyTag.id == RelationAgentTag.tag_id)\
                                            .filter(RelationAgentTag.agent_id == agent.id)
        agent_tags_list = list([tag_obj.value for tag_obj in agent_tags])
        agent.tags_ = ', '.join(agent_tags_list)
    session.close()
    return render_template('agents_main.html', agents=agents, tags=tags)


@bp.route('/<int:agent_id>', methods=['DELETE'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def agent_delete(agent_id):
    try:
        def rm_agent(s):
            s.query(DebbyAgent).filter(DebbyAgent.id == agent_id).delete()
            print('deleted agent. id: {}'.format(agent_id))
        lock_safe_query(rm_agent)
        # return redirect(url_for('agents.agents_main'))
        return jsonify({'status': 'ok'})
    except Exception as e:
        print('[!!!] agent_delete. Exception.')
        traceback.print_exc()
        abort(404)


@bp.route('/<int:agent_id>/edit', methods=['GET', 'POST'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def agent_edit(agent_id):

    session = new_session()
    dagent = session.query(DebbyAgent).filter(DebbyAgent.id == agent_id).first()
    known_tags = session.query(DebbyTag).all()
    currently_selected_tags = session.query(DebbyTag).filter(RelationAgentTag.tag_id == DebbyTag.id)\
                            .filter(RelationAgentTag.agent_id == dagent.id).all()
    session.close()

    if request.method == 'GET':

        for tag in known_tags:
            if tag in currently_selected_tags:
                tag.is_selected = True

        return render_template('agents_edit.html', agent=dagent, tags=known_tags)

    else:  # request.method == 'POST'
        try:

            VARS = validate_agent_data(request, known_tags)

            # Prepare updating data
            updating = {
                DebbyAgent.name: VARS['name'],
                DebbyAgent.address: VARS['address'],
                DebbyAgent.max_jobs: VARS['max_jobs']
            }
            if VARS['token']:
                updating[DebbyAgent.token] = VARS['token']

            session = new_session()
            session.query(DebbyAgent).filter(DebbyAgent.id == agent_id).update(updating)
            session.query(RelationAgentTag).filter(RelationAgentTag.agent_id == agent_id).delete()

            # Add new tags
            for tid in VARS['spec_tags_ids']:
                rat = RelationAgentTag(agent_id=agent_id, tag_id=tid)
                session.add(rat)

            session.commit()
            session.close()

            return redirect(url_for('agents.agents_main'))

        except DebbyValidateException as dex:
            return render_template('agents_edit.html', agent=dagent, tags=known_tags, error=dex.args)


@bp.route('/new', methods=['GET', 'POST'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def agents_new():

    session = new_session()
    known_tags = session.query(DebbyTag).all()
    session.close()

    if request.method == 'GET':
        return render_template('agents_new.html', tags=known_tags)

    else:  # POST
        try:
            VARS = validate_agent_data(request, known_tags)

            # create agent
            session = new_session()
            da = DebbyAgent(name=VARS['name'], address=VARS['address'], token=VARS['token'], max_jobs=VARS['max_jobs'])
            session.add(da)
            session.commit()

            # add agent-tag relations
            for tid in VARS['spec_tags_ids']:
                rat = RelationAgentTag(agent_id=da.id, tag_id=tid)
                session.add(rat)

            session.commit()
            session.close()

            return redirect(url_for('agents.agents_main'))

        except DebbyValidateException as dex:
            return render_template('agents_new.html', tags=known_tags, error=dex.args)


@bp.route('/tags/new', methods=['GET', 'POST'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def tag_new():

    if request.method == 'GET':
        return render_template('tag_new.html')

    else:  # POST
        try:
            # TODO: Validate user input
            value = request.form.get("tag_value")
            if len(value) == 0:
                raise DebbyValidateException('Incorrect tag value')

            dt = DebbyTag(value=value)
            session = new_session()
            session.add(dt)
            session.commit()
            session.close()

            return redirect(url_for('agents.agents_main'))
            
        except DebbyValidateException as dex:
            return render_template('agents_new.html', tags=known_tags, error=dex.args)


@bp.route('/tags/<int:tag_id>', methods=['DELETE'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def tag_delete(tag_id):

    session = new_session()
    session.query(DebbyTag).filter(DebbyTag.id == tag_id).delete()
    session.commit()
    session.close()

    return redirect(url_for('agents.agents_main'))
