

import requests
import traceback
import tenacity

from app.agents.base_client import AgentClient


class DebbyAgentClient(AgentClient):

    def __init__(self, agent):
        self.address = agent.address
        self.base_uri = self.address if "://" in self.address else "https://" + self.address
        self.api_url = self.base_uri + '/api/v1'

        self._AUTH_HEADER = {'X-AUTH-TOKEN': agent.token}

        self.STATUS_OK = 'ok'
        self.STATUS_ERROR = 'error'

    @tenacity.retry(wait=tenacity.wait_exponential(multiplier=1, min=4, max=45), stop=tenacity.stop_after_delay(360),
                    retry=tenacity.retry_if_result(lambda res: res and res.get('status') == 'error'),
                    retry_error_callback=(lambda retry_state: retry_state.outcome.result()))
    def _request(self, method, url, headers, json_=None):
        resp = None
        try:
            resp = method(url, headers=headers, json=json_, verify=False, timeout=10)
            return resp.json()
        except requests.ConnectionError:
            print('[!] DebbyAgentClient. _request. ConnectionError. url: {}'.format(url))
            return {'status': self.STATUS_ERROR, 'message': 'ConnectionError'}
        except ValueError:
            print('[!!!] DebbyAgentClient. _request. Exception on resp to json parsing. url: {}. json["engine"]: {}.'.format(url, json_.get("engine")))
            if resp:
                print('[!!!] DebbyAgentClient. _request. Exception on resp to json parsing. r.content: {}. r.headers: {}.'.format(resp.content, resp.headers))
            else:
                print('[!!!] DebbyAgentClient. _request. Exception on resp to json parsing. r.content: None')
            return {'status': self.STATUS_ERROR, 'message': ''}
        except Exception:
            traceback.print_exc()
            if resp:
                print('[!!!] DebbyAgentClient. _request. Exception. r.content: {}.'.format(resp.content))
            else:
                print('[!!!] DebbyAgentClient. _request. Exception. r.content: None')
            return {'status': self.STATUS_ERROR, 'message': ''}

    # def get_tasks_info(self):
    #     url = self.api_url + '/scans'
    #     res = self._request(requests.get, url, self._AUTH_HEADER)
    #     return res

    def get_task_info(self, task_uuid):
        """

        :param task_uuid: "9393123c-d103-4ff8-aede-1aeee38abca5"
        :return:
            finished:
                {
                  "status": "ok",
                  "task_info": {
                    "engine": "nmap",
                    "profile": "{u'args': u'...'}",                         # engine specific
                    "scan_stop_time": 1545668213,
                    "scan_start_time": 1545668211,
                    "state": "finished",
                    "entry_create_time": 1545668211,                        # redundant
                    "task_uuid": "9393123c-d103-4ff8-aede-1aeee38abca5",
                    "results": [...],                                       # engine specific
                    "save_to_db": false,
                    "entry_modify_time": 1545668213                         # redundant
                  }
                }
            OR
            in progress:
                {
                  "status": "ok",
                  "task_info": {
                    "engine": "nmap",
                    "profile": "{u'args': u'-sS 5.255.240.0/25'}",          # engine specific
                    "scan_stop_time": null,
                    "scan_start_time": 1545672279,
                    "state": "in progress",
                    "entry_create_time": 1545672279,                        # redundant
                    "task_uuid": "c8c31276-01d5-4b05-b57c-1dbd9b3e30d1",
                    "results": null,
                    "save_to_db": false,
                    "entry_modify_time": 1545672279                         # redundant
                  }
                }
        """
        url = self.api_url + '/scans/' + str(task_uuid)
        res = self._request(requests.get, url, self._AUTH_HEADER)
        return res

    def stop_task(self, task_uuid):
        """
        :param task_uuid: "9393123c-d103-4ff8-aede-1aeee38abca5"
        :return:
            {"status": "ok", "message": "scan was already finished"}
        """
        url = self.api_url + '/scans/' + str(task_uuid)
        res = self._request(requests.delete, url, self._AUTH_HEADER)
        return res

    def run_task(self, payload):
        """
        :param payload: {"engine": "nmap", "profile": {"args": "-sS 5.255.255.5/32"}, "save_to_db": false}
        :return:
            {'status': 'ok', 'task_uuid': 'b5e564b9-a6f8-4057-b000-4ca495d470d0'}
            or
            {'status': 'error, 'message': 'error'}

        """
        url = self.api_url + '/scans'
        # res = None
        # print('[+] DebbyAgentClient. run_task. Payload: {}'.format(payload))
        res = self._request(requests.post, url, self._AUTH_HEADER, json_=payload)
        return res