

import json
import requests
import tenacity
import traceback
import random

from app.agents.base_client import AgentClient
from app.settings import STATE_FINISHED, STATE_IN_PROGRESS, STATE_ABORTED, STATE_CANCELED
from app.settings import ENGINE_MOLLY
from app.settings import MOLLY_OAUTH_TOKEN, MOLLY_DEFAULT_AUTH_PROFILE


class MollyAgentClient(AgentClient):

    def __init__(self, agent):
        self.address = agent.address
        self.base_uri = "https://molly.yandex-team.ru"
        self.api_url = self.base_uri + '/api/v1.1'

        token = MOLLY_OAUTH_TOKEN or agent.token
        self._AUTH_HEADER = {'Authorization': 'OAuth {}'.format(token)}

        self.STATUS_OK = 'ok'
        self.STATUS_ERROR = 'error'

        self.ERROR_READING_REPORT_SUPBSTRING = "Error reading report"

    @tenacity.retry(wait=tenacity.wait_exponential(multiplier=1, min=4, max=60), stop=tenacity.stop_after_delay(300),
                    retry=tenacity.retry_if_result(lambda res: (
                        res.get('status') == 'error' and
                        "Error reading report" not in res.get("error", "") and
                        "Check if target is accessible from _SECURITY_MTN_NETS_" not in res.get("error", "")
                    )),
                    retry_error_callback=(lambda retry_state: retry_state.outcome.result()))
    def _request(self, method, url, headers, data=None):
        r = None
        try:
            r = method(url, headers=headers, data=data, verify=False)
            # print("[MOLLY] get_task_info. url: {}. r.text: {}.".format(url, r.text))
            return r.json()
        except requests.ConnectionError:
            return {'status': self.STATUS_ERROR, 'message': 'ConnectionError'}
        except Exception:
            if r:
                print("[!!!] MollyAgentClient. _request. EXCEPTION. r.text: {}".format(r.text))
            else:
                print("[!!!] MollyAgentClient. _request. EXCEPTION. NO RESPONSE")
            traceback.print_exc()
            return {'status': self.STATUS_ERROR, 'message': ''}

    # def get_tasks_info(self):
    #     url = self.api_url + '/scans'
    #     res = self._request(requests.get, url, self._AUTH_HEADER)
    #     return res

    def get_task_info(self, task_uuid):
        """
        :param task_uuid: "9393123c-d103-4ff8-aede-1aeee38abca5"
        :return:
            {
              "status": "ok",
              "task_info": {
                "engine": "nmap",
                "profile": "{u'args': u'...'}",
                "scan_stop_time": None,
                "scan_start_time": None,
                "state": "finished",
                "entry_create_time": None,                          # redundant
                "task_uuid": "9393123c-d103-4ff8-aede-1aeee38abca5",
                "results": {
                    "vulnerabilities" = [],
                    "report_url" = "",
                }
                "save_to_db": false,
                "entry_modify_time": None                           # redundant
              }
            }
        """
        url = self.api_url + '/scan/' + str(task_uuid)
        # print("[MOLLY] get_task_info. task_uuid: {}. url: {}".format(task_uuid, url))
        molly_resp = self._request(requests.get, url, self._AUTH_HEADER)
        # print("[MOLLY] get_task_info. task_uuid: {}. molly_resp: {}".format(task_uuid, molly_resp))

        # error_chance = random.randint(0, 9)
        # print("[MOLLY] get_task_info. task_uuid: {}. error_chance: {}".format(task_uuid, error_chance))
        # if not error_chance:
        #     return {
        #         'status': self.STATUS_ERROR,
        #         'message': 'unknown molly response status'
        #     }

        if molly_resp.get('status') == 'error':
            if self.ERROR_READING_REPORT_SUPBSTRING in molly_resp.get("error"):
                state = STATE_ABORTED
            else:
                state = STATE_CANCELED
            return {
                'status': self.STATUS_OK,
                'task_info': {
                    'engine': ENGINE_MOLLY,
                    'state': state,
                    'results': None,
                    "task_uuid": task_uuid
                }
            }
        elif molly_resp.get('status') == 'in_progress':
            return {
                'status': self.STATUS_OK,
                'task_info': {
                    'engine': ENGINE_MOLLY,
                    'state': STATE_IN_PROGRESS,
                    'results': None,
                    "task_uuid": task_uuid
                }
            }
        elif molly_resp.get('status') == 'done':
            return {
                'status': self.STATUS_OK,
                'task_info': {
                    'engine': ENGINE_MOLLY,
                    'state': STATE_FINISHED,
                    'results': {
                        'vulnerabilities': molly_resp.get('vulnerabilities'),
                        'report_url': molly_resp.get('report_url'),
                    },
                    "task_uuid": task_uuid
                }
            }
        else:
            return {
                'status': self.STATUS_ERROR,
                'message': 'unknown molly response status'
            }

    def stop_task(self, task_uuid):
        return None

    def run_task(self, payload):
        """
        :param payload:
            {
                "engine": "molly",
                "profile": {
                    "url": "https://yadi.yandex-team.ru/&https://yadi.yandex-team.ru/robots.txt",
                    "profile": "Crasher",
                    "resp": [],
                    "service": "CRASHER_yadi.yandex-team.ru"
                },
                "save_to_db": false
            }
        :return:
            {'status': 'ok', 'task_uuid': 'b5e564b9-a6f8-4057-b000-4ca495d470d0'}
            or
            {'status': 'error, 'message': 'error_msg'}
        """

        # print("[+] MollyAgentClient. payload: {}".format(payload))

        if not isinstance(payload, dict):
            payload = json.loads(payload)

        profile = payload.get('profile')
        url = self.api_url + '/scan/'
        params = {
            "profile": profile.get("profile", "Yandex"),
            "target_uri": profile.get('url'),
            "target": profile.get('service'),
            "users": ','.join(profile.get('resp')),
            "auth_profile": MOLLY_DEFAULT_AUTH_PROFILE,
            "is_prod": True,
        }

        # print("[+] MollyAgentClient. params: {}".format(params))

        # molly_resp = dict()
        molly_resp = self._request(requests.post, url, self._AUTH_HEADER, data=params)
        # print("[+] MollyAgentClient. payload: {}".format(payload))
        # print("[+] MollyAgentClient. molly_resp: {}".format(molly_resp))

        if molly_resp.get('status') != 'in_progress':
            return {
                'status': self.STATUS_ERROR,
                'message': ''
            }

        if len(molly_resp.get('scan_id')) > 1:
            print('[!] MOLLY RETURNED MANY SCAN_ID: {}'.format(molly_resp.get('scan_id')))

        return {
            'status': self.STATUS_OK,
            'task_uuid': molly_resp.get('scan_id')[0]
        }
