

import json
import zlib
import base64
import requests
import tenacity
import traceback

from app.agents.base_client import AgentClient
from app.settings import ENGINE_OPENVAS, STATE_PENDING
from app.settings import STATE_CANCELED, STATE_FINISHED, STATE_IN_PROGRESS
from app.utils import eprint


class OpenVASAgentClient(AgentClient):

    def __init__(self, agent):
        self.address = agent.address
        self.base_uri = self.address if "://" in self.address else "https://" + self.address
        self.api_url = self.base_uri

        self._AUTH_HEADER = {'X-OPENVAS-PASSWORD': agent.token}

        self.STATUS_OK = 'ok'
        self.STATUS_ERROR = 'error'

        self.OPENVAS_STATUS_OK = 'ok'
        self.OPENVAS_STATUS_ERROR = 'error'
        self.OPENVAS_STATUS_RUNNING = 'Running'
        self.OPENVAS_STATUS_STOP_REQUESTED = 'Stop Requested'
        self.OPENVAS_STATUS_STOPPED = 'Stopped'
        self.OPENVAS_STATUS_REQUESTED = 'Requested'
        self.OPENVAS_STATUS_QUEUED = 'Queued'
        self.OPENVAS_STATUS_DONE = 'Done'

    @tenacity.retry(wait=tenacity.wait_exponential(multiplier=1, min=4, max=60), stop=tenacity.stop_after_delay(300),
                    retry=tenacity.retry_if_result(lambda res: res.get('status') == 'error'),
                    retry_error_callback=(lambda retry_state: retry_state.outcome.result()))
    def _request(self, method, url, headers, data=None):
        r = None
        try:
            # print('[+] OpenVASAgentClient:_request. url: {}'.format(url))
            # if data:
            #     print('[+] OpenVASAgentClient:_request. len(data): {}'.format(len(data)))
            #     if len(data) < 100:
            #         print('[+] OpenVASAgentClient:_request. data: {}'.format(data))
            # eprint("{} {} {} {}".format(method, url, headers, data))
            # eprint(f"{method=} {url=} {headers=} {data=}")
            r = method(url, headers=headers, data=data, verify=False, timeout=10)
            # print('[+] OpenVASAgentClient:_request. r: {}'.format(r))
            return r.json()
        except requests.ConnectionError:
            eprint('[!] OpenVASAgentClient:_request. Exception: ConnectionError')
            return {'status': self.STATUS_ERROR, 'message': 'OpenVASAgentClient ConnectionError'}
        except ValueError:
            eprint("{}".format(r))
            eprint('[!] OpenVASAgentClient:_request. Exception: ValueError. Message: ResponseJSONDecodeError')
            resp_text = r.text if r else 'NO_RESPONSE_TEXT'
            return {'status': self.STATUS_ERROR, 'message': 'ResponseJSONDecodeError: {}'.format(resp_text)}
        except Exception as e:
            content = r.content if r else ''
            eprint('[!] OpenVASAgentClient:_request. Exception: {}. url: {}. r.content: {}.'.format(e, url, content))
            traceback.print_exc()
            return {'status': self.STATUS_ERROR, 'message': ''}

    def get_task_info(self, report_uuid):
        """
        :param report_uuid: "9393123c-d103-4ff8-aede-1aeee38abca5"
        :return:

        """
        url = self.api_url + '/report/' + str(report_uuid)
        resp = self._request(requests.get, url, self._AUTH_HEADER)
        resp_status = resp.get('status')

        # error status
        if resp_status != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': resp.get('message')}

        res = resp.get('res')
        task_status = res.get('status')

        # running
        if task_status in [self.OPENVAS_STATUS_RUNNING, self.OPENVAS_STATUS_STOP_REQUESTED]:
            return {
                'status': self.STATUS_OK,
                'task_info': {
                    'engine': ENGINE_OPENVAS,
                    'state': STATE_IN_PROGRESS,
                    'results': None,
                    "task_uuid": report_uuid
                }
            }

        # canceled/stoppped
        if task_status == self.OPENVAS_STATUS_STOPPED:
            return {
                'status': self.STATUS_OK,
                'task_info': {
                    'engine': ENGINE_OPENVAS,
                    'state': STATE_CANCELED,
                    'results': None,
                    "task_uuid": report_uuid
                }
            }

        # pending/in queue
        if task_status in [self.OPENVAS_STATUS_REQUESTED, self.OPENVAS_STATUS_QUEUED]:
            return {
                'status': self.STATUS_OK,
                'task_info': {
                    'engine': ENGINE_OPENVAS,
                    'state': STATE_PENDING,
                    'results': None,
                    "task_uuid": report_uuid
                }
            }

        # finished/done
        if task_status == self.OPENVAS_STATUS_DONE:
            return {
                'status': self.STATUS_OK,
                'task_info': {
                    'engine': ENGINE_OPENVAS,
                    'state': STATE_FINISHED,
                    'results': res.get('results'),
                    "task_uuid": report_uuid
                }
            }

        # Unknown behavior
        print('[!] Unknown OpenVAS response. report_uuid: {}. resp: {}'.format(report_uuid, resp))
        return {
            'status': self.STATUS_ERROR,
            'message': 'unknown openvas response'
        }

    def get_pdf_report(self, report_uuid):
        """
        :param report_uuid: "9393123c-d103-4ff8-aede-1aeee38abca5"
        :return:

        """
        url = self.api_url + '/pdf_report/' + str(report_uuid)
        resp = self._request(requests.get, url, self._AUTH_HEADER)

        # error status
        if resp.get('status') != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': resp.get('message')}

        res = resp.get('res')
        res = base64.b64decode(res.encode())
        res = zlib.decompress(res)
        res = base64.b64decode(res)

        return {'status': self.STATUS_OK, 'content': res}

    def stop_task(self, task_uuid):
        """
        :param task_uuid: "9393123c-d103-4ff8-aede-1aeee38abca5"
        :return:
            {"status": "ok", "message": "scan was already finished"}
        """

        url = self.api_url + '/task/' + str(task_uuid)
        resp = self._request(requests.delete, url, self._AUTH_HEADER)
        resp_status = resp.get('status')

        if resp_status == self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_OK, 'message': ''}
        else:
            return {'status': self.STATUS_ERROR, 'message': resp.get('message')}

    def run_task(self, payload):
        """
        :param payload:
            {
                "engine": "openvas",
                "profile": {
                    "task_uuid": "b5e564b9-a6f8-4057-b000-4ca495d470d0"
                },
                "save_to_db": false
            }
        :return:
            {'status': 'ok', 'task_uuid': 'b5e564b9-a6f8-4057-b000-4ca495d470d0'}
            or
            {'status': 'error, 'message': 'error_msg'}
        """

        if not isinstance(payload, dict):
            payload = json.loads(payload)

        profile = payload.get('profile')
        task_uuid = profile.get("task_uuid")  # this is openvas specific task_uuid
        url = self.api_url + '/task/{}'.format(task_uuid)
        resp = self._request(requests.post, url, self._AUTH_HEADER)

        if resp.get('status') != 'ok':
            return {
                'status': self.STATUS_ERROR,
                'message': resp.get('message')
            }

        return {
            'status': self.STATUS_OK,
            'task_uuid': resp.get('res').get('report_uuid')  # this is debby specific task_uuid
        }

    def export_scan_config(self, config_uuid):
        url = self.api_url + '/configs/{}'.format(config_uuid)
        resp = self._request(requests.get, url, self._AUTH_HEADER)
        resp_status = resp.get('status')

        if resp_status != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': resp.get("message")}

        return {"status": self.STATUS_OK, "config_str": resp.get('res')}

    def import_scan_config(self, config_str):
        url = self.api_url + '/configs'
        resp = self._request(requests.post, url, self._AUTH_HEADER, data=config_str)
        resp_status = resp.get('status')

        if resp_status != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': resp.get("message")}

        return {"status": self.STATUS_OK, "config_uuid": resp.get('res')}

    def create_port_list(self, port_list_str):
        url = self.api_url + '/port_list'
        resp = self._request(requests.post, url, self._AUTH_HEADER, data=port_list_str)
        resp_status = resp.get('status')

        if resp_status != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': resp.get("message")}

        return {"status": self.STATUS_OK, "port_list_uuid": resp.get('res')}

    def create_target(self, target_hosts_list, port_list_uuid):
        url = self.api_url + '/targets'
        if port_list_uuid:
            data = {"targets": ','.join(target_hosts_list), "port_list_uuid": port_list_uuid}
        else:
            data = {"targets": ','.join(target_hosts_list)}
        resp = self._request(requests.post, url, self._AUTH_HEADER, data=data)
        resp_status = resp.get('status')

        if resp_status != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': resp.get("message")}

        return {"status": self.STATUS_OK, "target_uuid": resp.get('res')}

    def create_task(self, target_uuid, config_uuid):
        url = self.api_url + '/tasks'
        data = {"target_uuid": target_uuid, "config_uuid": config_uuid}
        resp = self._request(requests.post, url, self._AUTH_HEADER, data=data)
        resp_status = resp.get('status')

        if resp_status != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': resp.get("message")}

        return {"status": self.STATUS_OK, "task_uuid": resp.get('res')}

    def delete_task(self, task_uuid):
        url = self.api_url + '/tasks/{}'.format(task_uuid)
        resp = self._request(requests.delete, url, self._AUTH_HEADER)
        resp_status = resp.get('status')

        if resp_status != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': resp.get("message")}

        return {"status": self.STATUS_OK}

    def delete_target(self, target_uuid):
        url = self.api_url + '/targets/{}'.format(target_uuid)
        resp = self._request(requests.delete, url, self._AUTH_HEADER)
        resp_status = resp.get('status')

        if resp_status != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': resp.get("message")}

        return {"status": self.STATUS_OK}

    def delete_port_list(self, port_list_uuid):
        url = self.api_url + '/port_list/{}'.format(port_list_uuid)
        resp = self._request(requests.delete, url, self._AUTH_HEADER)
        resp_status = resp.get('status')

        if resp_status != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': resp.get("message")}

        return {"status": self.STATUS_OK}

    def delete_config(self, config_uuid):
        url = self.api_url + '/configs/{}'.format(config_uuid)
        resp = self._request(requests.delete, url, self._AUTH_HEADER)
        resp_status = resp.get('status')

        if resp_status != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': resp.get("message")}

        return {"status": self.STATUS_OK}
