

import json
import requests
import tenacity
import traceback

from app.agents.base_client import AgentClient
from app.settings import ENGINE_OPENVAS, STATE_PENDING
from app.settings import STATE_CANCELED, STATE_FINISHED, STATE_IN_PROGRESS


class OpenVASAgentClient(AgentClient):

    def __init__(self, agent):
        self.address = agent.address
        self.base_uri = "https://" + self.address
        self.api_url = self.base_uri

        self._AUTH_HEADER = {'X-OPENVAS-PASSWORD': agent.token}

        self.STATUS_OK = 'ok'
        self.STATUS_ERROR = 'error'

        self.OPENVAS_STATUS_OK = 'ok'
        self.OPENVAS_STATUS_ERROR = 'error'
        self.OPENVAS_STATUS_RUNNING = 'Running'
        self.OPENVAS_STATUS_STOP_REQUESTED = 'Stop Requested'
        self.OPENVAS_STATUS_STOPPED = 'Stopped'
        self.OPENVAS_STATUS_REQUESTED = 'Requested'
        self.OPENVAS_STATUS_DONE = 'Done'

    @tenacity.retry(wait=tenacity.wait_exponential(multiplier=1, min=4, max=60), stop=tenacity.stop_after_delay(300),
                    retry=tenacity.retry_if_result(lambda res: res.get('status') == 'error'),
                    retry_error_callback=(lambda retry_state: retry_state.outcome.result()))
    def _request(self, method, url, headers):
        r = None
        try:
            r = method(url, headers=headers, verify=False)
            return r.json()
        except requests.ConnectionError:
            return {'status': self.STATUS_ERROR, 'message': 'OpenVASAgentClient ConnectionError'}
        except Exception as e:
            content = r.content if r else ''
            print('[!] OpenVASAgentClient. Exception: {}. r.content: {}.'.format(e, r.content))
            traceback.print_exc()
            return {'status': self.STATUS_ERROR, 'message': ''}

    def get_task_info(self, report_uuid):
        """
        :param report_uuid: "9393123c-d103-4ff8-aede-1aeee38abca5"
        :return:

        """
        url = self.api_url + '/report/' + str(report_uuid)
        resp = self._request(requests.get, url, self._AUTH_HEADER)
        resp_status = resp.get('status')

        # error status
        if resp_status != self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_ERROR, 'message': 'openvas error status'}

        res = resp.get('res')
        task_status = res.get('status')

        # running
        if task_status in [self.OPENVAS_STATUS_RUNNING, self.OPENVAS_STATUS_STOP_REQUESTED]:
            return {
                'status': self.STATUS_OK,
                'task_info': {
                    'engine': ENGINE_OPENVAS,
                    'state': STATE_IN_PROGRESS,
                    'results': None,
                    "task_uuid": report_uuid
                }
            }

        # canceled/stoppped
        if task_status == self.OPENVAS_STATUS_STOPPED:
            return {
                'status': self.STATUS_OK,
                'task_info': {
                    'engine': ENGINE_OPENVAS,
                    'state': STATE_CANCELED,
                    'results': None,
                    "task_uuid": report_uuid
                }
            }

        # pending/in queue
        if task_status == self.OPENVAS_STATUS_REQUESTED:
            return {
                'status': self.STATUS_OK,
                'task_info': {
                    'engine': ENGINE_OPENVAS,
                    'state': STATE_PENDING,
                    'results': None,
                    "task_uuid": report_uuid
                }
            }

        # finished/done
        if task_status == self.OPENVAS_STATUS_DONE:
            return {
                'status': self.STATUS_OK,
                'task_info': {
                    'engine': ENGINE_OPENVAS,
                    'state': STATE_FINISHED,
                    'results': res.get('results'),
                    "task_uuid": report_uuid
                }
            }

        # Unknown behavior
        print('[!] Unknown OpenVAS response: {}'.format(resp))
        return {
            'status': self.STATUS_ERROR,
            'message': 'unknown openvas response'
        }

    def stop_task(self, task_uuid):
        """
        :param task_uuid: "9393123c-d103-4ff8-aede-1aeee38abca5"
        :return:
            {"status": "ok", "message": "scan was already finished"}
        """

        url = self.api_url + '/task/' + str(task_uuid)
        resp = self._request(requests.delete, url, self._AUTH_HEADER)
        resp_status = resp.get('status')

        if resp_status == self.OPENVAS_STATUS_OK:
            return {'status': self.STATUS_OK, 'message': ''}
        else:
            return {'status': self.STATUS_ERROR, 'message': ''}

    def run_task(self, payload):
        """
        :param payload:
            {
                "engine": "openvas",
                "profile": {
                    "task_uuid": "b5e564b9-a6f8-4057-b000-4ca495d470d0"
                },
                "save_to_db": false
            }
        :return:
            {'status': 'ok', 'task_uuid': 'b5e564b9-a6f8-4057-b000-4ca495d470d0'}
            or
            {'status': 'error, 'message': 'error_msg'}
        """

        if not isinstance(payload, dict):
            payload = json.loads(payload)

        profile = payload.get('profile')
        task_uuid = profile.get("task_uuid")  # this is openvas specific task_uuid

        url = self.api_url + '/task/{}'.format(task_uuid)
        resp = self._request(requests.post, url, self._AUTH_HEADER)

        # print("[+] OpenVASAgentClient. run_task. url: {}. resp: {}".format(url, resp))

        if resp.get('status') != 'ok':
            return {
                'status': self.STATUS_ERROR,
                'message': ''
            }

        return {
            'status': self.STATUS_OK,
            'task_uuid': resp.get('res').get('report_uuid')  # this is debby specific task_uuid
        }
