

import traceback
from flask import Blueprint, jsonify, request, g
from werkzeug.exceptions import BadRequest

from app.agents.utils import get_api_available_agents
from app.policies.utils import create_policy, get_policy_by_uuid
from app.projects.utils import create_projectapiinfo, check_projects_limit_for_api_src_id
from app.projects.utils import create_new_project
from app.projects.utils import get_projects_by_api_src_id
from app.projects.utils import delete_project_by_id_and_api_src_id
from app.projects.utils import get_project_info_by_id_and_api_src_id
from app.api.v1.validators import validate_api_new_project_params
from app.scans.utils import check_api_src_permission_for_scan, get_scan_details
from app.validators import DebbyValidateException, DebbyRuntimeException, DebbyOwnerException
from app.yauth import oauth_scope_required, user_group_required
from app.settings import PUBLIC_API_USERS, PRIVATE_API_USERS, ENGINE_NMAP
from app import yauth


bp = Blueprint('api_v1', __name__, url_prefix='/api/v1')


@bp.route('/projects', methods=['POST'])
@oauth_scope_required(['debby:init'])
# @user_group_required(PRIVATE_API_USERS)
@yauth.require_role(yauth.ROLES_PRIVATE_API)
def create_project():
    api_src_id = g.yauid

    try:
        # validate params. but dont change values
        params = validate_api_new_project_params(request.get_json())

        check_projects_limit_for_api_src_id(api_src_id)

        # create policy
        new_policy_id = create_policy(params.get('name'), params.get('transport'), ports_list=params.get('ports_list'),
                                      additional_options=params.get('additional_options'))

        # create
        new_project_id = create_new_project(params.get('name'), params.get('targets'), params.get('engine'),
                                            new_policy_id, params.get('start_timestamp'), params.get('stop_timestamp'),
                                            params.get('scan_interval'), params.get('tags_list'), list(),
                                            max_scan_time=params.get('max_scan_time'), log_closed=params.get('log_closed'))

        # info about project owner
        create_projectapiinfo(new_project_id, api_src_id, False)

        return jsonify({"status": "ok", "result": {"project": {"id": new_project_id}}})

    except BadRequest:
        return jsonify({"status": "error", "message": "Incorrect request data"})

    except DebbyValidateException as e:
        return jsonify({"status": "error", "message": "Validation Error: {}".format(e.args)}), 400

    except:
        traceback.print_exc()
        return jsonify({"status": "error", "message": "Internal Error"}), 500


@bp.route('/projects', methods=['GET'])
@oauth_scope_required(['debby:read'])
def get_projects():
    api_src_id = g.yauid
    try:
        projects = get_projects_by_api_src_id(api_src_id)
        return jsonify({"status": "ok", "result": {"projects": projects}})

    except:
        traceback.print_exc()
        return jsonify({"status": "error", "message": "Internal Error"}), 500


@bp.route('/projects/<int:project_id>', methods=['GET'])
@oauth_scope_required(['debby:read'])
def get_project_info(project_id):
    api_src_id = g.yauid
    try:
        get_targets = request.args.get('targets') == '1'
        get_policy_ = request.args.get('policy') == '1'
        project_info = get_project_info_by_id_and_api_src_id(project_id, api_src_id, get_targets, get_policy_)
        return jsonify({"status": "ok", "result": {"project": project_info}})

    except DebbyRuntimeException as e:
        return jsonify({"status": "error", "message": e.args}), 400

    except DebbyOwnerException as e:
        return jsonify({"status": "error", "message": e.args}), 403

    except:
        traceback.print_exc()
        return jsonify({"status": "error", "message": "Internal Error"}), 500


@bp.route('/projects/<int:project_id>', methods=['DELETE'])
@oauth_scope_required(['debby:init'])
def delete_project(project_id):
    api_src_id = g.yauid
    try:
        delete_project_by_id_and_api_src_id(project_id, api_src_id)
        return jsonify({"status": "ok"}), 200

    except DebbyOwnerException as e:
        return jsonify({"status": "error", "message": e.args}), 403

    except DebbyRuntimeException as e:
        return jsonify({"status": "error", "message": e.args}), 400

    except:
        traceback.print_exc()
        return jsonify({"status": "error", "message": "Internal Error"}), 500


@bp.route('/scans/<int:scan_id>', methods=['GET'])
@oauth_scope_required(['debby:init'])
def get_scan_results(scan_id):
    api_src_id = g.yauid
    try:
        get_results = request.args.get('results') == '1'
        check_api_src_permission_for_scan(api_src_id, scan_id)
        scan_details = get_scan_details(scan_id, get_results)
        return jsonify({"status": "ok", "result": scan_details})

    except DebbyRuntimeException as e:
        return jsonify({"status": "error", "message": e.args}), 404

    except DebbyOwnerException as e:
        return jsonify({"status": "error", "message": e.args}), 403

    except:
        traceback.print_exc()
        return jsonify({"status": "error", "message": "Internal Error"}), 500


@bp.route('/agents', methods=['GET'])
@oauth_scope_required(['debby:init'])
def get_available_agents():

    try:
        agents_list = get_api_available_agents()
        return jsonify({"status": "ok", "result": agents_list})

    except:
        traceback.print_exc()
        return jsonify({"status": "error", "message": "Internal Error"}), 500


@bp.route('/projects-public', methods=['POST'])
@oauth_scope_required(['debby:init'])
# @user_group_required(PUBLIC_API_USERS)
@yauth.require_role(yauth.ROLES_PUBLIC_API)
def create_project_public():
    api_src_id = g.yauid

    try:
        # validate params. but dont change values
        params = validate_api_new_project_params(request.get_json(), policy_by_uuid=True)

        check_projects_limit_for_api_src_id(api_src_id)

        # get policy
        policy = get_policy_by_uuid(params.get('policy_uuid'))
        if not policy and params.get('engine') == ENGINE_NMAP:
            raise DebbyValidateException('Unknown policy uuid specified for nmap engine')
        policy_id = policy.id if policy else None

        # create
        new_project_id = create_new_project(params.get('name'), params.get('targets'), params.get('engine'),
                                            policy_id, params.get('start_timestamp'), params.get('stop_timestamp'),
                                            params.get('scan_interval'), params.get('tags_list'), list(),
                                            log_closed=params.get('log_closed'))

        # info about project owner
        create_projectapiinfo(new_project_id, api_src_id, True)

        return jsonify({"status": "ok", "result": {"project": {"id": new_project_id}}})

    except BadRequest:
        return jsonify({"status": "error", "message": "Incorrect request data"})

    except DebbyValidateException as e:
        return jsonify({"status": "error", "message": "Validation Error: {}".format(e.args)}), 400

    except:
        traceback.print_exc()
        return jsonify({"status": "error", "message": "Internal Error"}), 500
