

import datetime
import functools
from flask import request, jsonify, g

from app.agents.utils import get_known_tags, create_tag_if_not_exists
# from app.settings import DEBBY_API_TVM_CLIENTS
from app.settings import ENGINE_NMAP, PROTO_TCP
from app.settings import TRANSPORT_PROTOCOLS
from app.settings import API_TAG, API_ENGINES, API_PREFIX
from app.validators import DebbyValidateException, check_port, prepare_project_targets
from app.utils import now as utils_now, timestamp_utc_2_datetime_msk
# from app.tvm import check_tvm_ticket


# def handle_tvm_ticket(func):
#
#     @functools.wraps(func)
#     def wrapper(*args, **kwargs):
#         src_id = None
#
#         if DONT_CHECK_SERVICE_TICKETS:
#             src_id = 1
#
#         # TVM
#         if not src_id:
#             srv_ticket = request.headers.get('X-Ya-Service-Ticket')
#             if srv_ticket:
#                 src_id = check_tvm_ticket(srv_ticket)
#                 if src_id and src_id not in DEBBY_API_TVM_CLIENTS:
#                     return jsonify({"status": "error", "message": "Permission Denied"})
#
#         if not src_id:
#             return jsonify({"status": "error", "message": "Authentication Check Error"})
#
#         g.api_src_id = str(src_id)
#         return func(*args, **kwargs)
#
#     return wrapper


def validate_api_new_project_params(data, policy_by_uuid=False):
    """
        Check params of request with content-type: application/json.
        Return dict of params if correct.
        Otherwise raises exception DebbyValidateException
    :raise:
        DebbyValidateException - incorrect data on validation. Details in message.
    :param data:
        json object with params
    :return:
        dict of validated params
    """

    # ENGINE
    engine = data.get('engine')
    if not isinstance(engine, str) and not isinstance(engine, str):
        raise DebbyValidateException('Incorrect name format')
    if engine not in API_ENGINES:
        raise DebbyValidateException('Unsupported engine')

    # NAME
    name = data.get('name')
    if name is not None:
        if not isinstance(name, str) and not isinstance(name, str):
            raise DebbyValidateException('Incorrect name format')
        if len(name) == 0 or len(name) > 128:
            raise DebbyValidateException('Name is too long')
    # prepare some api values for names and tags
    if not name.startswith(API_PREFIX):
        name = API_PREFIX + name

    # TARGETS
    targets = data.get('targets')
    if not isinstance(targets, list):
        if not isinstance(targets, str) and not isinstance(targets, str):
            raise DebbyValidateException('Incorrect targets format')
        targets = prepare_project_targets(targets)
    if len(targets) == 0:
        raise DebbyValidateException('Empty targets')

    # START TIMESTAMP
    start_ts = data.get('start_timestamp')
    if start_ts is not None:
        try:
            start_ts = int(start_ts)
        except:
            raise DebbyValidateException("Incorrect start_timestamp format")

    # INTERVAL
    scan_interval = data.get('scan_interval')
    if scan_interval is not None:
        try:
            scan_interval = int(scan_interval)
        except:
            raise DebbyValidateException("Incorrect interval format")

    # max_scan_time
    max_scan_time = data.get('max_scan_time')
    if max_scan_time is not None:
        try:
            max_scan_time = int(max_scan_time)
        except:
            raise DebbyValidateException("Incorrect interval format")

    # STOP TIMESTAMP
    stop_timestamp = data.get('stop_timestamp')
    if stop_timestamp is not None:
        try:
            stop_timestamp = int(stop_timestamp)
        except:
            raise DebbyValidateException("Incorrect stop_timestamp format")

    if not start_ts:
        start_ts = utils_now()
        scan_interval = timestamp_utc_2_datetime_msk(int(0))
        stop_timestamp = start_ts + datetime.timedelta(hours=1)
    else:
        if start_ts and stop_timestamp and scan_interval and start_ts < stop_timestamp and scan_interval >= 0:
            start_ts = timestamp_utc_2_datetime_msk(start_ts)
            scan_interval = timestamp_utc_2_datetime_msk(scan_interval)
            stop_timestamp = timestamp_utc_2_datetime_msk(stop_timestamp)
        else:
            raise DebbyValidateException("Incorrect time or interval format")

    # ADDITIONAL OPTIONS
    additional_options = data.get('additional_options')
    if (additional_options is not None and
            not isinstance(additional_options, str) and
            not isinstance(additional_options, str)):
        raise DebbyValidateException("Incorrect scripts format")

    # ----
    # TAGS
    # ----
    tags_list = data.get('tags')
    if tags_list is None:
        tags_list = list()
    elif not isinstance(tags_list, list):
        raise DebbyValidateException("Incorrect tags format")
    # add default API tag
    tags_list.append(API_TAG)
    # create api tag in if not exists
    create_tag_if_not_exists(API_TAG)
    # get all known tags
    known_tags = get_known_tags()
    # remain only known tag ids
    tag_ids = set()
    for tag in tags_list:
        if not isinstance(tag, str) and not isinstance(tag, str):
            raise DebbyValidateException("Incorrect tags format")
        for known_tag in known_tags:
            if known_tag.value == tag:
                tag_ids.add(known_tag.id)
    # set -> list
    tag_ids_list = list(tag_ids)

    # ----

    policy_uuid = None
    transport = None
    ports = None

    if policy_by_uuid:
        policy_uuid = data.get('policy_uuid')
        if not policy_uuid and engine == ENGINE_NMAP:
            raise DebbyValidateException("Policy uuid not specified. This param is required for nmap engine.")
    else:
        # TRANSPORT - nmap specific
        if engine != ENGINE_NMAP:
            transport = PROTO_TCP
        else:
            transport = data.get('transport')
            if transport not in TRANSPORT_PROTOCOLS:
                raise DebbyValidateException("Incorrect transport specified for nmap engine")

        # PORTS - nmap specific
        if engine != ENGINE_NMAP:
            ports = None
        else:
            ports = data.get('ports')
            if ports is not None:
                if not isinstance(ports, str) and not isinstance(ports, str):
                    raise DebbyValidateException('Incorrect ports format. String excpected')
                if ports:
                    ports = ports.replace(' ', '').split(',')
                    if not all([check_port(p) for p in ports]):
                        raise DebbyValidateException('Incorrect ports')

    # LOG_CLOSED
    log_closed = data.get('log_closed') is True

    return {
        # general
        'engine': engine,
        'name': name,
        'targets': targets,
        'log_closed': log_closed,

        # agent tags
        'tags_list': tag_ids_list,

        # engine options
        'additional_options': additional_options,

        # scheduler
        'start_timestamp': start_ts,
        'scan_interval': scan_interval,
        'stop_timestamp': stop_timestamp,

        # nmap specific
        'ports_list': ports,
        'transport': transport,
        'policy_uuid': policy_uuid,
        'max_scan_time': max_scan_time
    }
