

import tenacity
import psycopg2
import traceback

from ilock import ILock
from sqlalchemy import exc
from sqlalchemy import create_engine
from sqlalchemy.orm import sessionmaker

from app.utils import eprint
from .models import Base
from app.settings import DB_URLS, DB_CONNECT_ARGS, LOCK_AGENTS

DB_CURRENT_MASTER_IDX = 0
engine = None
Session = None


def _update_engine_and_sessionmaker():
    global engine
    global Session
    global DB_CURRENT_MASTER_IDX

    if not DB_CONNECT_ARGS:
        engine = create_engine(DB_URLS[DB_CURRENT_MASTER_IDX], pool_recycle=7200, 
                               pool_size=15, max_overflow=95, echo=False)
    else:
        engine = create_engine(DB_URLS[DB_CURRENT_MASTER_IDX], pool_recycle=7200, 
                               pool_size=15, max_overflow=95, connect_args=DB_CONNECT_ARGS, echo=False)

    Session = sessionmaker(bind=engine)


@tenacity.retry(reraise=True, wait=tenacity.wait_fixed(1),
                stop=tenacity.stop_after_attempt(1000), retry=tenacity.retry_if_result(lambda value: value is False))
def select_writable_instance():
    global engine
    global DB_CURRENT_MASTER_IDX

    for i in range(DB_CURRENT_MASTER_IDX, DB_CURRENT_MASTER_IDX + len(DB_URLS)):
        if not engine:
            _update_engine_and_sessionmaker()

        try:
            res = engine.execute('SHOW transaction_read_only')
            if res.first()[0] == 'off':
                return True
        except Exception:
            traceback.print_exc()

        DB_CURRENT_MASTER_IDX = (DB_CURRENT_MASTER_IDX + 1) % len(DB_URLS)
        _update_engine_and_sessionmaker()

    else:
        print('[!][DB] Did not found writable instance!!!!')
        return False


# def select_writable_instance(retries=100):
#     global DB_CURRENT_MASTER_IDX
#     global engine
#     global Session
#
#     while retries > 0:
#
#         try:
#             for i in range(DB_CURRENT_MASTER_IDX, DB_CURRENT_MASTER_IDX+len(DB_URLS)):
#                 # print('[+][DB] Trying ID: {}.'.format(DB_CURRENT_MASTER_IDX))
#                 if not engine:
#                     _update_engine_and_sessionmaker()
#
#                 try:
#                     res = engine.execute('SHOW transaction_read_only')
#                     if res.first()[0] == 'off':
#                         break
#                 except psycopg2.OperationalError:
#                     pass
#
#                 # print('[+][DB] Changing on ID: {}.'.format(DB_CURRENT_MASTER_IDX))
#                 DB_CURRENT_MASTER_IDX = (DB_CURRENT_MASTER_IDX+1) % len(DB_URLS)
#                 _update_engine_and_sessionmaker()
#
#             else:
#                 print('[+][DB] Did not found writable instance!!!!')
#
#             return
#
#         except Exception as e:  # psycopg2.OperationalError
#             traceback.print_exc()
#             if retries == 0:
#                 raise e
#
#         time.sleep(1)
#         retries -= 1


# ------- #
# Session #
# ------- #


def new_session():
    """ Creates new sqlalchemy session """
    select_writable_instance()
    return Session()


def get_engine():
    return engine


# -------------------------- #
# Init, Drop, Refresh tables #
# -------------------------- #


def init_tables():
    select_writable_instance()
    Base.metadata.create_all(engine)


def drop_tables():
    select_writable_instance()
    Base.metadata.reflect(engine)
    Base.metadata.drop_all(engine)


def refresh_tables():
    drop_tables()
    init_tables()

# -----------------
# --- Decorator ---
# -----------------


@tenacity.retry(reraise=True, stop=tenacity.stop_after_delay(60*24*5), wait=tenacity.wait_fixed(2))
def safe_query(foo, *args, **kwargs):

    session = new_session()
    session.expire_on_commit = False

    try:
        res = foo(session, *args)
        session.commit()
        session.close()
        return res

    except (exc.SQLAlchemyError, psycopg2.OperationalError) as e:
        traceback.print_exc()
        session.rollback()
        session.close()
        raise e


def lock_safe_query(func):
    @tenacity.retry(wait=tenacity.wait_random(min=1, max=3), stop=tenacity.stop_after_attempt(5))
    def lock_query():
        with ILock(LOCK_AGENTS, timeout=15):
            try:
                safe_query(func)
            except:
                print('[!!!] lock_safe_query. EXCEPTION!!!')
                traceback.print_exc()

    try:
        lock_query()
    except:
        print('[!!!] cancel_scan. lock_query. not ok')
        traceback.print_exc()
        safe_query(func)
