import uuid

from sqlalchemy import Column, Integer, String, ForeignKey, SmallInteger
from sqlalchemy import Boolean, DateTime, TIMESTAMP
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.ext.declarative import declarative_base

from app.utils import now as utils_now
from app.settings import BULK_STATUS_INACTIVE


Base = declarative_base()


class DebbyPolicy(Base):
    __tablename__ = 'debbypolicy'

    id = Column(Integer, primary_key=True)
    name = Column(String, nullable=False)
    ports = Column(String, nullable=False)
    scan_type = Column(String, nullable=False)
    create_time = Column(DateTime, default=utils_now)
    uuid = Column(String, default=(lambda: str(uuid.uuid4())))

    def __repr__(self):
        return "<DebbyPolicy(id={}, name='{}', ports='{}', scan_type='{}')>".format(
            self.id, self.name, self.ports, self.scan_type)


class DebbyPolicyScript(Base):
    __tablename__ = 'debbypolicyscripts'

    id = Column(Integer, primary_key=True)
    policy_id = Column(Integer, ForeignKey(DebbyPolicy.id, ondelete="CASCADE"), nullable=False)
    name = Column(String, nullable=False)
    args = Column(String)


class DebbyPolicyAdditionalOptions(Base):
    __tablename__ = 'debbypolicyadditionaloptions'

    policy_id = Column(Integer, ForeignKey(DebbyPolicy.id, ondelete="CASCADE"), primary_key=True)
    value = Column(String)


class DebbyTag(Base):
    __tablename__ = 'debbytag'

    id = Column(Integer, primary_key=True)
    value = Column(String, nullable=False)
    create_time = Column(DateTime, default=utils_now)

    def __repr__(self):
        return "<DebbyATag(id={}, value='{}')>".format(self.id, self.value)   


class DebbyAgent(Base):
    __tablename__ = 'debbyagent'

    id = Column(Integer, primary_key=True)
    name = Column(String, nullable=False)
    address = Column(String, nullable=False)
    token = Column(String, nullable=False)
    jobs = Column(Integer, nullable=False, default=0)
    max_jobs = Column(Integer, nullable=False, default=0)

    def __repr__(self):
        return "<DebbyAgent(id={}, name='{}', address='{}', token='{}', jobs='{}')>".format(
            self.id, self.name, self.address, 
            self.token[:5] + '*' * 10 + self.token[-5:], self.jobs)
        

class RelationAgentTag(Base):
    __tablename__ = 'relationagenttag'

    id = Column(Integer, primary_key=True)
    create_time = Column(DateTime, default=utils_now)
    agent_id = Column(Integer, ForeignKey(DebbyAgent.id, ondelete="CASCADE"))
    tag_id = Column(Integer, ForeignKey(DebbyTag.id, ondelete="CASCADE"))


class Namespace(Base):
    __tablename__ = 'namespace'

    id = Column(Integer, primary_key=True)
    name = Column(String)
    contacts = Column(String)

    def __repr__(self):
        return "<Namespace(id={}, name={})>".format(self.id, self.name)


class DebbyProject(Base):
    __tablename__ = 'DebbyProject'

    id = Column(Integer, primary_key=True)
    policy_id = Column(Integer, ForeignKey(DebbyPolicy.id, ondelete="SET NULL"))
    name = Column(String, nullable=False)
    engine = Column(String, nullable=False)
    targets = Column(String, nullable=False)
    create_time = Column(DateTime, default=utils_now)
    scan_start = Column(DateTime, nullable=False)
    interval = Column(TIMESTAMP, nullable=False)
    max_scan_time = Column(Integer)
    scan_stop = Column(DateTime)
    last_scan = Column(DateTime)
    paused = Column(Boolean, nullable=False, default=False)
    deleted = Column(Boolean, nullable=False, default=False)
    save_to_db = Column(Boolean, nullable=False, default=False)
    log_closed = Column(Boolean, nullable=False, default=False)  # send info about closed ports to splunk
    st_ticket = Column(String)
    retries = Column(Integer, nullable=False, default=0)
    retry_period = Column(Integer, nullable=False, default=0)
    contacts = Column(String, nullable=False, default="")
    namespace_id = Column(Integer, ForeignKey(Namespace.id, ondelete="SET NULL"))

    def __repr__(self):
        return "<DebbyProject(id={}, name='{}', engine='{}', paused={})>".format(
            self.id, self.name, self.engine, self.paused)

class RelationProjectTag(Base):
    __tablename__ = 'RelationProjectTag'

    id = Column(Integer, primary_key=True)
    create_time = Column(DateTime, default=utils_now)
    project_id = Column(Integer, ForeignKey(DebbyProject.id, ondelete="CASCADE"))
    tag_id = Column(Integer, ForeignKey(DebbyTag.id, ondelete="CASCADE"))


class DebbyScan(Base):
    __tablename__ = 'debbyscan'

    id = Column(Integer, primary_key=True)
    project_id = Column(Integer, ForeignKey(DebbyProject.id, ondelete="SET NULL"))
    create_time = Column(DateTime, default=utils_now)
    finish_time = Column(DateTime)
    log_finish_time = Column(DateTime)
    state = Column(String, nullable=False)
    all_targets_tasked = Column(Boolean, nullable=False, default=False)
    # scanned_targets = Column(Integer, nullable=False, default=0)
    # total_targets = Column(Integer, nullable=False, default=0)

    def __repr__(self):
        return "<DebbyScan(id={}, project_id='{}', create_time='{}')>".format(
            self.id, self.project_id, self.create_time)


class DebbyTask(Base):
    __tablename__ = 'debbytask'

    id = Column(Integer, primary_key=True)
    debbyscan_id = Column(Integer, ForeignKey(DebbyScan.id, ondelete="CASCADE"))
    debbyagent_id = Column(Integer, ForeignKey(DebbyAgent.id, ondelete="SET NULL"))
    task_uuid = Column(UUID(as_uuid=True))
    targets = Column(String)
    state = Column(String)
    create_time = Column(DateTime, default=utils_now)
    finish_time = Column(DateTime, default=None)
    sent_to_analysis = Column(DateTime)
    sent_logs = Column(Boolean, nullable=False, default=False)

    def __repr__(self):
        return "<DebbyTask(id={}, debbyscan_id='{}', task_uuid='{}', targets='{}', status='{}')>".format(
            self.id, self.debbyscan_id, self.task_uuid, self.targets, self.state)


class IPAddressBulk(Base):
    __tablename__ = 'ipaddressbulk'

    id = Column(Integer, primary_key=True)
    status = Column(String, nullable=False, default=BULK_STATUS_INACTIVE)
    update_time = Column(DateTime)


class IPAddress(Base):
    __tablename__ = 'ipaddress'

    id = Column(Integer, primary_key=True)
    address = Column(String, nullable=False)
    protocol = Column(String, nullable=False)
    bulk_id = Column(Integer, ForeignKey(IPAddressBulk.id, ondelete="CASCADE"))

    def __repr__(self):
        return "<IPAddress(id={}, address='{}', protocol='{}')>".format(
            self.id, self.address, self.protocol)


class Macro(Base):
    __tablename__ = 'macro'

    id = Column(Integer, primary_key=True)
    value = Column(String, nullable=False)
    targets = Column(String, nullable=False)
    status = Column(String, nullable=False, default=BULK_STATUS_INACTIVE)
    update_time = Column(DateTime)


# ----------------
# --- Pipeline ---
# ----------------


class PipelineProjects(Base):
    __tablename__ = 'pipelineprojects'

    id = Column(Integer, primary_key=True)
    prev_proj_id = Column(Integer, ForeignKey(DebbyProject.id, ondelete="CASCADE"))
    next_proj_id = Column(Integer, ForeignKey(DebbyProject.id, ondelete="CASCADE"))


class PipelineScans(Base):
    __tablename__ = 'pipelinescans'

    id = Column(Integer, primary_key=True)
    prev_scan_id = Column(Integer, ForeignKey(DebbyScan.id, ondelete="CASCADE"))
    next_scan_id = Column(Integer, ForeignKey(DebbyScan.id, ondelete="CASCADE"))
    state = Column(String)


class DebbyScanResults(Base):
    __tablename__ = 'scanresults'

    id = Column(Integer, primary_key=True)
    scan_id = Column(Integer, ForeignKey(DebbyScan.id, ondelete="CASCADE"))
    ip = Column(String)
    port = Column(Integer)
    transport = Column(String)
    enabled = Column(Boolean, default=True)
    time = Column(TIMESTAMP)
    state = Column(String, default='')


class DebbyScanResultsScripts(Base):
    __tablename__ = 'scanresultsscripts'

    id = Column(Integer, primary_key=True)
    scan_result_id = Column(Integer, ForeignKey(DebbyScanResults.id, ondelete="CASCADE"))
    key = Column(String)
    value = Column(String)


class DebbyScanResultsService(Base):
    __tablename__ = 'scanresultsservices'

    id = Column(Integer, primary_key=True)
    scan_result_id = Column(Integer, ForeignKey(DebbyScanResults.id, ondelete="CASCADE"))
    name = Column(String)
    product = Column(String)
    version = Column(String)


class ProjectApiInfo(Base):
    __tablename__ = 'projectapiinfo'

    id = Column(Integer, primary_key=True)
    project_id = Column(Integer, ForeignKey(DebbyProject.id, ondelete="CASCADE"))
    api_src_id = Column(String)
    is_public = Column(Boolean, nullable=False, default=False)

# KeyValue for MetaTable


class MetaTable(Base):
    __tablename__ = 'metatable'

    key = Column(String, primary_key=True)
    value = Column(String)


class UnistatTable(Base):
    __tablename__ = 'unistattable'

    key = Column(String, primary_key=True)
    value = Column(Integer)


class PortCache(Base):
    __tablename__ = 'portcache'

    target = Column(String, primary_key=True)       # ipv4/ipv6/(fqdn?)
    port = Column(Integer, primary_key=True)
    location = Column(SmallInteger, primary_key=True)    # internal/external/guest/ycloud/...
    transport = Column(SmallInteger, primary_key=True)    # tcp/udp
    last_seen = Column(DateTime, nullable=False)
    info_source = Column(String)


class RetryTargets(Base):
    __tablename__ = 'retrytargets'

    id = Column(Integer, primary_key=True)
    scan_id = Column(Integer, ForeignKey(DebbyScan.id, ondelete="CASCADE"), index=1)
    failed_task_id = Column(Integer, ForeignKey(DebbyTask.id, ondelete="CASCADE"))
    cur_task_id = Column(Integer, ForeignKey(DebbyTask.id, ondelete="CASCADE"), index=1)
    failed_count = Column(Integer, nullable=False, default=1)

    def __repr__(self):
        return "<RetryTargets(id={}, scan_id={}, failed_task_id={}, cur_task_id={}, failed_count={})>".format(
            self.id, self.scan_id, self.failed_task_id, self.cur_task_id, self.failed_count)
