

import json

from app.engines.base_engine import BaseEngine
from app.db.db import new_session
from app.db.models import DebbyProject, DebbyPolicy, PipelineScans, DebbyScanResults
from app.db.models import DebbyTask, DebbyScan, DebbyTag, RelationProjectTag
from app.engines.utils import target_generator_helper
from app.resps_resolver import CachedRespResolver
from app.settings import PROTO_TCP
from app.utils import addr_to_proto, datetime_msk_2_timestamp_utc, is_valid_ipv4_address, is_valid_ipv6_address
from app.utils import AppSecDiscoveryClient


class ClickHouseCheckerEngine(BaseEngine):

    @staticmethod
    def _prepare_profile(policy, targetlist):

        targets = ','.join("{}:{}".format(t[0], t[1]) for t in targetlist)
        args = '-t ' + targets
        profile = {"args": args}

        return profile

    @staticmethod
    def _get_targets_from_db(prev_scan_id):
        session = new_session()
        targets = session.query(DebbyScanResults.ip, DebbyScanResults.port)\
                         .filter(DebbyScanResults.scan_id == prev_scan_id)\
                         .filter(DebbyScanResults.enabled == True).distinct()
        target_list = [(ip, port) for (ip, port) in targets]
        session.close()

        targets = list()
        for ip, port in target_list:
            if is_valid_ipv4_address(ip):
                targets.append((ip, port))
            elif is_valid_ipv6_address(ip):
                targets.append(("[{}]".format(ip), port))

        return targets

    @staticmethod
    def _get_targets_from_project(project_id):
        targets = list()

        s = new_session()
        project = s.query(DebbyProject).filter(DebbyProject.id == project_id).first()
        s.close()

        targets_ = project.targets.split(', ')

        for t in targets_:
            splitted = t.strip().rsplit(':', 1)

            if len(splitted) == 2 and splitted[1].isdigit():
                port = int(splitted[1])
            else:
                port = 8123

            target = splitted[0]

            if target[0] == '[' and target[-1] == ']':
                target = target[1:-1]

            for ip in target_generator_helper([target]):
                if is_valid_ipv4_address(ip):
                    targets.append((ip, port))
                elif is_valid_ipv6_address(ip):
                    targets.append(("[{}]".format(ip), port))

        return targets

    @staticmethod
    def new_tasks_payloads_generator(project_id, scan_id=None):
        s = new_session()
        project = s.query(DebbyProject).filter(DebbyProject.id == project_id).first()
        s.close()

        targets = list()
        if scan_id:
            session = new_session()
            scan_pipeline = session.query(PipelineScans).filter(PipelineScans.next_scan_id == scan_id).first()
            session.close()
            if scan_pipeline:
                targets = ClickHouseCheckerEngine._get_targets_from_db(scan_pipeline.prev_scan_id)
            else:
                targets = None

        if targets is None:
            targets = ClickHouseCheckerEngine._get_targets_from_project(project_id)

        total = len(targets)
        cnt = 0
        while cnt < total:

            some_targets = targets[cnt:cnt+128]

            payload = json.dumps({
                "engine": project.engine,
                "profile": ClickHouseCheckerEngine._prepare_profile(None, some_targets),
                "save_to_db": project.save_to_db
            })

            targets_list = list()
            for t in some_targets:
                targets_list.append("{}:{}".format(t[0], t[1]))

            yield (targets_list, payload)

            cnt += 128

        return

    @staticmethod
    def scan_results_to_splunk_events(scan_results, task_id, only_enabled=False):
        """

        :param results:
        :param task_id:
        :param scan:
        :param project:
        :param policy:
        :return:
        """

        events = list()
        cached_resolver = CachedRespResolver()

        session = new_session()
        task = session.query(DebbyTask).filter(DebbyTask.id == task_id).first()
        scan = session.query(DebbyScan).filter(DebbyScan.id == task.debbyscan_id).first()
        project = session.query(DebbyProject).filter(DebbyProject.id == scan.project_id).first()
        policy = session.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).first()
        tags = session.query(DebbyTag).filter(RelationProjectTag.project_id == project.id)\
                                      .filter(RelationProjectTag.tag_id == DebbyTag.id).all()
        session.close()

        tag_list = list([tag.value for tag in tags])

        for res in scan_results:
            addr = res['target']
            enabled = res['enabled']
            port = res['port']

            if not enabled and only_enabled:
                continue

            if enabled:
                scripts = {
                    'clickhousechecker_default_auth': res['default_auth'],
                    'clickhousechecker_readonly_auth': res['readonly_auth'],
                    'clickhousechecker_version': res['version']
                }
            else:
                scripts = None

            resp, source = cached_resolver.get_resps(addr)
            resp = ','.join(resp)

            events.append({
                'event_type': 'info',

                'projectId': project.id,
                'projectName': project.name,
                'engine': project.engine,
                'logClosed': project.log_closed,
                'tags': tag_list,

                'policyId': policy.id,

                'dest_ip': addr,
                'resp': resp,
                'resp_source': source,
                'protocol': addr_to_proto(addr),

                'dest_port': port,
                'transport': PROTO_TCP,
                'time': None,
                'enabled': enabled,
                'scripts': scripts,

                'service_name': None,
                'service_product': None,
                'service_version': None,

                'scanId': scan.id,
                'scanStartTime': datetime_msk_2_timestamp_utc(scan.create_time),

                'taskId': task.id,
            })

        return events
