

import json
from sqlalchemy import desc

from app.agents.agentcli import new_agent_client
from app.engines.base_engine import BaseEngine
from app.db.db import new_session
from app.db.models import DebbyProject, DebbyPolicy, DebbyPolicyAdditionalOptions, DebbyTask, DebbyScan, DebbyAgent
from app.db.models import RelationProjectTag, DebbyTag
# from app.projects.utils import _get_projects_engine_by_id
from app.settings import PROTO_TCP
from app.tasks.utils import get_available_agent_2
from app.validators import DebbyException
from app.utils import build_splunk_event, is_valid_ipv6_address, is_valid_ipv4_address, addr_to_proto
from app.utils import datetime_msk_2_timestamp_utc
from app.engines.meta_info import MetaInfo
from app.utils import eprint
from app.target_resolver import TargetResolver, ResultingTargetType
from app.settings import STATE_FINISHED
from app.resps_resolver import CachedRespResolver


def _get_projects_engine_by_id(project_id):
    s = new_session()
    project = s.query(DebbyProject).filter(DebbyProject.id == project_id).first()
    s.close()
    return project.engine


def split_on_master_and_slaves(agents):
    if len(agents) == 0:
        return None
    if len(agents) == 1:
        return agents[0], list()

    master_idx = 0
    for idx in range(len(agents)):
        if 'MASTER' in agents[idx].name:
            master_idx = idx
            break

    master = agents[master_idx]
    slaves = agents[:master_idx] + agents[master_idx+1:]
    return master, slaves


class OpenvasEngine(BaseEngine):

    @staticmethod
    def scan_init(scan_id, required_tags_list, project_id, *args, **kwargs):
        """
            Makes copy of config from master on all agents (slaves and create copy on master)

        :param required_tags_list:
        :param project_id:
        :param args:
        :param kwargs:
        :return:
        """
        session = new_session()
        project = session.query(DebbyProject).filter(DebbyProject.id == project_id).first()
        policy = session.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).first()
        additional_options = session.query(DebbyPolicyAdditionalOptions) \
                                    .filter(DebbyPolicyAdditionalOptions.policy_id == policy.id).first()

        config_uuid = additional_options.value
        session.close()

        agents = get_available_agent_2(required_tags_list, any_job_count=True, get_all=True)
        if not agents:
            return None

        master, slaves = split_on_master_and_slaves(agents)

        # master_client = new_agent_client(master)
        # # get master config
        # res = master_client.export_scan_config(config_uuid)
        # if res.get('status') != master_client.STATUS_OK:
        #     raise DebbyException("Export master config error")
        # config_str = res.get('config_str')
        # # make copy of config on master
        # # res = master_client.import_scan_config(config_str)
        # # if res.get('status') != master_client.STATUS_OK:
        # #     raise DebbyException("Import config to master error")
        # # master_config_uuid = res.get('config_uuid')

        # # save meta_data
        master_config_uuid = config_uuid
        MetaInfo.set("meta_openvas_scan_{}_agent_{}".format(scan_id, master.id), "{}".format(master_config_uuid))

        # for slave in slaves:
        #     # import config to all slave agents
        #     slave_client = new_agent_client(slave)
        #     res = slave_client.import_scan_config(config_str)
        #     if res.get('status') != slave_client.STATUS_OK:
        #         raise DebbyException("Import config to slave {} error".format(slave.address))
        #     slave_config_uuid = res.get('config_uuid')
        #     # save meta info for slave
        #     MetaInfo.set("meta_openvas_scan_{}_agent_{}".format(scan_id, slave.id), "{}".format(slave_config_uuid))

    @staticmethod
    def task_init(scan_id, agent, payload, targets, *args, **kwargs):
        session = new_session()
        scan = session.query(DebbyScan).filter(DebbyScan.id == scan_id).first()
        project = session.query(DebbyProject).filter(DebbyProject.id == scan.project_id).first()
        policy = session.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).first()
        session.close()

        agent_client = new_agent_client(agent)

        # prepare port_list
        ports_str = policy.ports.strip().replace(" ", "")
        if ports_str:
            if policy.scan_type == PROTO_TCP:
                ports_str = "T:" + ports_str
            else:
                ports_str = "U:" + ports_str

            # create port_list on agent
            res = agent_client.create_port_list(ports_str)
            if res.get('status') != agent_client.STATUS_OK:
                raise DebbyException("Error on create port_list on slave {}. msg: {}.".format(agent_client.address, res.get('message')))
            port_list_uuid = res.get("port_list_uuid")
        else:
            port_list_uuid = None

        # create_target
        res = agent_client.create_target(targets, port_list_uuid)
        if res.get('status') != agent_client.STATUS_OK:
            raise DebbyException("Error on create target on slave {}. msg: {}.".format(agent_client.address, res.get('message')))
        target_uuid = res.get("target_uuid")

        # get config uuid for agent from meta_info table
        config_uuid = MetaInfo.get("meta_openvas_scan_{}_agent_{}".format(scan_id, agent.id))

        # create_task
        res = agent_client.create_task(target_uuid, config_uuid)
        if res.get('status') != agent_client.STATUS_OK:
            raise DebbyException("Error on create task on slave {}. msg: {}.".format(agent_client.address, res.get('message')))
        task_uuid = res.get("task_uuid")

        # build meta_info
        task_meta_info = {"port_list_uuid": port_list_uuid, "task_uuid": task_uuid, "target_uuid": target_uuid}
        return task_meta_info

    @staticmethod
    def task_pre_run(payload, task_meta_info, *args, **kwargs):
        payload["profile"].update({"task_uuid": task_meta_info.get('task_uuid')})
        return payload

    @staticmethod
    def task_post_run(debby_task_id, task_meta_info, *args, **kwargs):
        MetaInfo.set("meta_openvas_task_{}".format(debby_task_id), json.dumps(task_meta_info))

    @staticmethod
    def scan_cleanup(scan_id, required_tags_list, *args, **kwargs):
        agents = get_available_agent_2(required_tags_list, any_job_count=True, get_all=True)
        if not agents:
            return None

        for agent in agents:
            agent_client = new_agent_client(agent)
            config_uuid = MetaInfo.get("meta_openvas_scan_{}_agent_{}".format(scan_id, agent.id))
            # agent_client.delete_config(config_uuid)
            MetaInfo.delete("meta_openvas_scan_{}_agent_{}".format(scan_id, agent.id))

    @staticmethod
    def task_cleanup(task, *args, **kwargs):
        # session = new_session()
        # agent = session.query(DebbyAgent).filter(DebbyAgent.id == task.debbyagent_id).first()
        # session.close()

        meta_info_key = "meta_openvas_task_{}".format(task.id)
        # task_meta_info = json.loads(MetaInfo.get(meta_info_key))
        # agent_client = new_agent_client(agent)

        # agent_client.delete_task(task_meta_info.get("task_uuid"))
        # agent_client.delete_target(task_meta_info.get("target_uuid"))
        # if task_meta_info.get("port_list_uuid"):
        #     agent_client.delete_port_list(task_meta_info.get("port_list_uuid"))

        MetaInfo.delete(meta_info_key)

    @staticmethod
    def _prepare_profile(policy):
        """

        :param policy:
        :return:
            {"task_uuid": policy.additional_options.value}
        """

        session = new_session()
        additional_options = session.query(DebbyPolicyAdditionalOptions)\
                                    .filter(DebbyPolicyAdditionalOptions.policy_id == policy.id).first()
        session.close()

        if not additional_options:
            print('[!] OpenvasEngine. _prepare_profile. Additional options must be specified')
            raise DebbyException('Additional Options are not specified')

        profile = {"task_uuid": additional_options.value}

        return profile

    @staticmethod
    def _prepare_targets(targets, policy, project_engine=None):
        """

        :param target_list:
        :param project_engine:
        :return:
        """

        payload = {
            "engine": project_engine,
            "profile": OpenvasEngine._prepare_profile(policy),
            "save_to_db": False
        }

        return targets, payload

    @staticmethod
    def new_tasks_payloads_generator(project_id, scan_id=None):
        s = new_session()
        project = s.query(DebbyProject).filter(DebbyProject.id == project_id).first()
        policy = s.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).first()
        s.close()

        # dummy target
        targets_raw = project.targets.split(',')
        all_targets_generator = TargetResolver.resolve_many_to_ip_or_fqdn(targets_raw, ResultingTargetType.IP_OR_FQDN)

        # project engine
        project_engine = _get_projects_engine_by_id(project_id)

        current_amount_of_taragets_per_task = 1
        RECALC_PERIOD = 0

        while True:
            if RECALC_PERIOD == 10:
                s = new_session()
                tasks = s.query(DebbyTask).filter(DebbyTask.debbyscan_id == scan_id)\
                                          .filter(DebbyTask.finish_time != None)\
                                          .filter(DebbyTask.state == STATE_FINISHED)\
                                          .order_by(desc(DebbyTask.id)).limit(10)
                s.close()

                only_tasks_with_finish_time = list([t for t in tasks if t.finish_time])
                tasks = only_tasks_with_finish_time
                if len(tasks) > 5:
                    task_targets_cntr = 0
                    task_time_cntr = 0
                    for task in tasks:
                        task_targets_cntr += len(task.targets.split(' '))
                        diff_time = task.finish_time - task.create_time
                        task_time_cntr += int(diff_time.total_seconds())

                    avg_targets = float(task_targets_cntr) / len(tasks)
                    avg_time = float(task_time_cntr) / len(tasks)
                    wanted_avg_time = 60*15

                    multiplier = float(wanted_avg_time) / avg_time
                    if multiplier > 10:
                        multiplier = 10
                    if multiplier < -10:
                        multiplier = -10

                    recalced_avg_targets = int(multiplier * avg_targets)
                    current_amount_of_taragets_per_task = recalced_avg_targets if recalced_avg_targets > 1 else 1

                    # dbg_info_ = "tasks: {}. avg_targets: {}. avg_time: {}. wanted_avg_time: {}. recalced_avg_targets: {}."
                    # dbg_info = dbg_info_.format(len(tasks), avg_targets, avg_time, wanted_avg_time, recalced_avg_targets)
                    # print ('[+] OpenvasEngine. new_tasks_payloads_generator. Recalc dbg: {}'.format(dbg_info))

                RECALC_PERIOD = 0

            RECALC_PERIOD += 1

            no_more_targets = False
            next_task_targets = list()
            for _ in range(current_amount_of_taragets_per_task):
                try:
                    next_task_targets.append(next(all_targets_generator))
                except StopIteration:
                    no_more_targets = True
                    break

            if next_task_targets:
                targets = next_task_targets
                targets, payload = OpenvasEngine._prepare_targets(targets, policy, project_engine)
                yield (targets, payload)

            if no_more_targets:
                break

    @staticmethod
    def scan_results_to_splunk_events(scan_results, task_id, only_enabled=True):

        cached_resolver = CachedRespResolver()

        splunk_events = list()

        session = new_session()
        task = session.query(DebbyTask).filter(DebbyTask.id == task_id).first()
        scan = session.query(DebbyScan).filter(DebbyScan.id == task.debbyscan_id).first()
        project = session.query(DebbyProject).filter(DebbyProject.id == scan.project_id).first()
        policy = session.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).first()
        tags = session.query(DebbyTag).filter(RelationProjectTag.project_id == project.id)\
                                      .filter(RelationProjectTag.tag_id == DebbyTag.id).all()
        session.close()

        tag_list = list([tag.value for tag in tags])

        # print('[+] scan_results_to_splunk_events. scan_results: {}.'.format(scan_results))

        for scan_result in scan_results:

            dest_ip = scan_result.get('host')
            dest_host = scan_result.get('hostname')

            if dest_ip:
                protocol = addr_to_proto(dest_ip)
            else:
                protocol = None

            dest_port = scan_result.get('port_id') if scan_result.get('port_id') >= 0 else 'general'

            openvas_scripts = {
                'ov_description': scan_result.get('description'),
                'ov_name': scan_result.get('name'),
                'ov_nvt': scan_result.get('nvt'),
                'ov_qod': scan_result.get('qod'),
                'ov_qod_type': scan_result.get('qod_type'),
                'ov_severity': scan_result.get('severity'),
                'ov_threat': scan_result.get('threat'),
            }

            # get resps if non log severity
            resp = list()
            resp_source = None
            if float(openvas_scripts['ov_severity']) > 0:
                resp, resp_source = cached_resolver.get_resps(dest_host)
                if not resp:
                    resp, resp_source = cached_resolver.get_resps(dest_ip)

            event = build_splunk_event(
                event_type='info',
                projectId=project.id, projectName=project.name, engine=project.engine,
                logClosed=project.log_closed, tags=tag_list,
                policyId=policy.id,
                dest_ip=dest_ip, dest_host=dest_host, resp=','.join(resp), resp_source=resp_source, protocol=protocol,
                dest_port=dest_port, transport=scan_result.get('transport'), time=None, enabled=True,

                scripts=openvas_scripts,

                service_name=None, service_product=None, service_version=None,
                scanId=scan.id, scanStartTime=datetime_msk_2_timestamp_utc(scan.create_time),
                taskId=task.id,
            )
            splunk_events.append(event)

            # print('[+] scan_results_to_splunk_events. event: {}.'.format(event))

        # print('[+] scan_results_to_splunk_events. splunk_events: {}.'.format(splunk_events))

        return splunk_events
