

from app.engines.base_engine import BaseEngine
from app.db.db import new_session
from app.db.models import DebbyProject, DebbyPolicy, DebbyPolicyAdditionalOptions, DebbyTask, DebbyScan
from app.db.models import RelationProjectTag, DebbyTag
from app.validators import DebbyException
from app.utils import build_splunk_event, is_valid_ipv6_address, is_valid_ipv4_address, addr_to_proto
from app.utils import datetime_msk_2_timestamp_utc


class OpenvasEngine(BaseEngine):

    @staticmethod
    def _prepare_profile(policy):
        """

        :param policy:
        :return:
            {"task_uuid": policy.additional_options.value}
        """

        session = new_session()
        additional_options = session.query(DebbyPolicyAdditionalOptions)\
                                    .filter(DebbyPolicyAdditionalOptions.policy_id == policy.id).first()
        session.close()

        if not additional_options:
            print('[!] OpenvasEngine. _prepare_profile. Additional options must be specified')
            raise DebbyException('Additional Options are not specified')

        profile = {"task_uuid": additional_options.value}

        return profile

    @staticmethod
    def _get_projects_engine_by_id(project_id):
        """

        :param project_id:
        :return:
            OPENVAS
        """
        s = new_session()
        project = s.query(DebbyProject).filter(DebbyProject.id == project_id).first()
        s.close()
        return project.engine

    @staticmethod
    def _prepare_target(target, policy, project_engine=None):
        """

        :param target_list:
        :param project_engine:
        :return:
        """

        payload = {
            "engine": project_engine,
            "profile": OpenvasEngine._prepare_profile(policy),
            "save_to_db": False
        }

        return target, payload

    @staticmethod
    def new_tasks_payloads_generator(project_id, scan_id=None):
        s = new_session()
        project = s.query(DebbyProject).filter(DebbyProject.id == project_id).first()
        policy = s.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).first()
        s.close()

        # dummy target
        target = '_'

        # project engine
        project_engine = OpenvasEngine._get_projects_engine_by_id(project_id)

        target, payload = OpenvasEngine._prepare_target(target, policy, project_engine)
        yield (target, payload)

    @staticmethod
    def scan_results_to_splunk_events(scan_results, task_id, only_enabled=True):
        splunk_events = list()

        session = new_session()
        task = session.query(DebbyTask).filter(DebbyTask.id == task_id).first()
        scan = session.query(DebbyScan).filter(DebbyScan.id == task.debbyscan_id).first()
        project = session.query(DebbyProject).filter(DebbyProject.id == scan.project_id).first()
        policy = session.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).first()
        tags = session.query(DebbyTag).filter(RelationProjectTag.project_id == project.id)\
                                      .filter(RelationProjectTag.tag_id == DebbyTag.id).all()
        session.close()

        tag_list = list([tag.value for tag in tags])

        print('[+] scan_results_to_splunk_events. scan_results: {}.'.format(scan_results))

        for scan_result in scan_results:

            dest_host = scan_result.get('host')
            if is_valid_ipv4_address(dest_host) or is_valid_ipv6_address(dest_host):
                dest_ip = dest_host
            else:
                dest_ip = None

            if dest_ip:
                protocol = addr_to_proto(dest_ip)
            else:
                protocol = None

            dest_port = scan_result.get('port_id') if scan_result.get('port_id') >= 0 else 'general'

            openvas_scripts = {
                'ov_description': scan_result.get('description'),
                'ov_name': scan_result.get('name'),
                'ov_nvt': scan_result.get('nvt'),
                'ov_qod': scan_result.get('qod'),
                'ov_qod_type': scan_result.get('qod_type'),
                'ov_severity': scan_result.get('severity'),
                'ov_threat': scan_result.get('threat'),
            }

            event = build_splunk_event(
                event_type='info',
                projectId=project.id, projectName=project.name, engine=project.engine,
                logClosed=project.log_closed, tags=tag_list,
                policyId=policy.id,
                dest_ip=dest_ip, dest_host=dest_host, resp=None, protocol=protocol,
                dest_port=dest_port, transport=scan_result.get('transport'), time=None, enabled=True,

                scripts=openvas_scripts,

                service_name=None, service_product=None, service_version=None,
                scanId=scan.id, scanStartTime=datetime_msk_2_timestamp_utc(scan.create_time),
                taskId=task.id,
            )
            splunk_events.append(event)

            print('[+] scan_results_to_splunk_events. event: {}.'.format(event))

        print('[+] scan_results_to_splunk_events. splunk_events: {}.'.format(splunk_events))

        return splunk_events
