import re
import time
import datetime
import ipaddress
import traceback
import logging
from dns.resolver import NXDOMAIN, NoAnswer, Resolver, LifetimeTimeout

from app.ivre_utils import net2range, ip2int, int2ip, range2nets
from app.macros import MacroBaseObject
from app.settings import MAX_TARGETS_PER_TASK_NMAP
from app.utils import is_valid_ipv4_address, is_valid_ipv6_address, is_subnet
from app.db.db import new_session
from app.db.models import PortCache
from app import portcache


def list_of_ip_to_subnet(target_list):
    last = None

    for target in target_list:
        target_int = ip2int(target)[0]

        if last and target_int != last + 1:
            return target_list

        last = target_int

    nets = range2nets((target_list[0], target_list[-1]))

    # return nets
    if len(nets) == 1:
        return nets
    else:
        return target_list

# ------------------------
# --- Target generator ---
# ------------------------

def taget_generator_to_ip_and_subnets(targets):

    ips = list()
    nets = list()

    for target in targets:

        if is_valid_ipv6_address(target) or is_valid_ipv4_address(target):
            ips.append(ipaddress.ip_address(target))

        elif is_subnet(target):
            prefix = target.split('/')[-1]
            start, _ = net2range(target)
            # start = ip2int(start)[0]
            trgt = start + '/' + prefix
            nets.append(ipaddress.ip_network(trgt))

        else:
            # macro
            macro = MacroBaseObject(target)
            resolved_targets = macro.get_targets()

            # ipv4, ipv6, subnets
            if resolved_targets:
                for resolved_target in resolved_targets:
                    # resolved to ip4, ip6 or subnet
                    if is_valid_ipv6_address(resolved_target) or is_valid_ipv4_address(resolved_target):
                        ips.append(ipaddress.ip_address(resolved_target))

                    elif is_subnet(resolved_target):
                        nets.append(ipaddress.ip_network(resolved_target))
                    # resolved to dns name
                    else:
                        for t in target_generator_for_domain_name(resolved_target):
                            ips.append(ipaddress.ip_address(t))
            # dns
            else:
                for t in target_generator_for_domain_name(target):
                    ips.append(ipaddress.ip_address(t))

    return (ips, nets)


def is_ip_or_subnet(value):
    if is_valid_ipv4_address(value) or is_valid_ipv6_address(value) or is_subnet(value):
        return True
    else:
        return False

        
def target_generator_for_debby_cache(value):
    # ex: "_DEBBY_CACHE_EXTERNAL_443_TCP_20D_"
    m = re.match("_DEBBY_CACHE_([^_\s]*)_(\d*)_([^_\s]*)_(\d+)(W|D|M)_", value)
    if not m:
        return
    
    location, port, transport, offset_value, offset_factor = m.groups()
    port = int(port)
    offset_value = int(offset_value)
    # transport = transport.lower()
    transport = portcache.transport2code(transport)
    location = portcache.location2code(location)

    offset_time = datetime.datetime.utcnow()
    if offset_factor == "M":
        offset_time -= datetime.timedelta(minutes=offset_value)
    elif offset_factor == "D":
        offset_time -= datetime.timedelta(days=offset_value)
    elif offset_factor == "W":
        offset_time -= datetime.timedelta(weeks=offset_value)

    s = new_session()
    results = s.query(PortCache.target).filter(PortCache.location == location).filter(PortCache.port == port).filter(PortCache.transport == transport).filter(PortCache.last_seen > offset_time).all()
    s.close()

    print("[+] target_generator_for_debby_cache. location={}, port={}, transport={}, offset_value={}, offset_factor={}, results={}".format(
        location, port, transport, offset_value, offset_factor, len(results)))

    for r in results:
        yield r.target

    return


def target_generator_for_ip_or_subnet(value):
    # IPv4
    if is_valid_ipv4_address(value):
        yield value
        return

    # IPv6
    if is_valid_ipv6_address(value):
        yield value
        return

    # IPv4 or IPv6 subnet
    start, stop = net2range(value)
    start = ip2int(start)[0]
    stop = ip2int(stop)[0]
    while start <= stop:
        yield (int2ip(start))
        start += 1


def target_generator_for_domain_name(value):
    """
    input = 'ya.ru'
    output = ['87.250.250.242', '2a02:6b8::2:242']

    :param value:
    :return:
    """
    # print('[+] target_generator_for_domain_name. target_list: {}'.format(value))

    # suspect to me a macros. no dns rescords
    if value.endswith("_"):
        return

    try:
        dns_resolver = Resolver()
    except NXDOMAIN:
        return

    try:
        ipv4_answer = dns_resolver.query(value, "A")
    except (NoAnswer, NXDOMAIN, LifetimeTimeout) as e:
        ipv4_answer = []
    for ipv4 in list([str(x) for x in ipv4_answer]):
        yield ipv4

    try:
        ipv6_answer = dns_resolver.query(value, "AAAA")
    except (NoAnswer, NXDOMAIN, LifetimeTimeout) as e:
        ipv6_answer = []
    for ipv6 in list([str(x) for x in ipv6_answer]):
        yield ipv6


def target_generator_helper(target_list):
    """
    Input:
        1)IPv4 Address/Subnet,
        2)IPv6 Address/Subnet,
        3)Macro,
        4)DNS Name
        5)Macro with overwrited behavior like _DEBBY_CACHE_...
    Output:
        IPv4 or IPv6 Address
    Example:
        Input:
            ['127.0.0.1', '::1', 'somedomain.ru', '_MACRO_TEST_', 192.168.0.0/31]
        Output:
            ['127.0.0.1', '::1', '11.22.33.44', '2000::', '192.168.0.0', '192.168.0.1']
    """

    for target in target_list:

        # ip or subnet
        if is_ip_or_subnet(target):
            for t in target_generator_for_ip_or_subnet(target):
                yield t

        # debby prescans cache
        # target = "_DEBBY_CACHE_EXTERNAL_443_TCP_2W_"
        elif target.strip().startswith("_DEBBY_CACHE_"):
            for t in target_generator_for_debby_cache(target.strip()):
                yield t

        # Macros or DNS
        else:
            # macro
            macro = MacroBaseObject(target)
            resolved_targets = macro.get_targets()
            # ipv4, ipv6, subnets
            if resolved_targets:
                for resolved_target in resolved_targets:
                    # resolved to ip4, ip6 or subnet
                    if is_ip_or_subnet(resolved_target):
                        for t in target_generator_for_ip_or_subnet(resolved_target):
                            yield t
                    # resolved to dns name
                    else:
                        for t in target_generator_for_domain_name(resolved_target):
                            yield t
            # dns
            else:
                for t in target_generator_for_domain_name(target):
                    yield t


def is_exclude(target, exc_ips, exc_nets):
    """

    :param target:
        string representing ip address
    :param exc_ips:
        list of ipaddress.ip_addresss objects
    :param exc_nets:
        list of ipaddress.ip_network objects

    :return:
        True/False
    """
    try:
        ip = ipaddress.ip_address(target)

        # filter by ip
        if ip in exc_ips:
            return True

        # filter by network
        for exc_net in exc_nets:
            if ip in exc_net:
                return True

    except:
        traceback.print_exc()

    return False


def target_generator(targetlist, excludelist=None):
    """
    Input:
        1)IPv4 Address/Subnet,
        2)IPv6 Address/Subnet,
        3)Macro,
        4)DNS Name,
        5)Macro with overwrited behavior like _DEBBY_CACHE_...
    Output:
        List of elements.
        Each elemnt is a list of ipv4 or ipv6 targets
    Example:
        Input:
            ['127.0.0.1', '::1', 'somedomain.ru', '_MACRO_TEST_', 192.168.0.0/31]
        Output:
            [
                ['127.0.0.1', '11.22.33.44', '192.168.0.0', '192.168.0.1'],
                ['::1', '2000::']
            ]
    """

    # print('[+] target_generator. targetlist: {}'.format(targetlist))
    max_targets = MAX_TARGETS_PER_TASK_NMAP

    ipv4_targets = list()
    ipv6_targets = list()

    if not excludelist or not isinstance(excludelist, list):
        excludelist = list()

    print("[+] target_generator. taget_generator_to_ip_and_subnets -> {}.".format(excludelist))
    exclude_ips, exclude_nets = taget_generator_to_ip_and_subnets(excludelist)
    print("[+] target_generator. taget_generator_to_ip_and_subnets -> {} -> {}".format(excludelist, (exclude_ips, exclude_nets)))

    # iterate for targets
    for target in target_generator_helper(targetlist):

        if is_exclude(target, exclude_ips, exclude_nets):
            continue

        if is_valid_ipv4_address(target):
            ipv4_targets.append(target)
            if len(ipv4_targets) == max_targets:
                yield ipv4_targets
                ipv4_targets = list()

        elif is_valid_ipv6_address(target):
            ipv6_targets.append(target)
            if len(ipv6_targets) == max_targets:
                yield ipv6_targets
                ipv6_targets = list()

    if len(ipv4_targets) > 0:
        yield ipv4_targets
    if len(ipv6_targets) > 0:
        yield ipv6_targets

