

import json

from app.engines.base_engine import BaseEngine
from app.db.db import new_session
from app.db.models import DebbyProject, DebbyPolicy, PipelineScans
from app.db.models import DebbyScanResults, DebbyPolicyAdditionalOptions
from app.db.models import DebbyTask, DebbyScan, DebbyTag, RelationProjectTag
from app.engines.utils import target_generator_helper
from app.resps_resolver import CachedRespResolver
from app.settings import PROTO_TCP
from app.utils import addr_to_proto, datetime_msk_2_timestamp_utc


class ZookeeperCheckerEngine(BaseEngine):

    @staticmethod
    def _prepare_profile(policy, targetlist):
        args_list = list()
        targets = ','.join(targetlist)
        args_list.append("-t {}".format(targets))

        session = new_session()
        additional_options = session.query(DebbyPolicyAdditionalOptions) \
                                    .filter(DebbyPolicyAdditionalOptions.policy_id == policy.id).first()
        session.close()
        if additional_options and additional_options.value:
            args_list.append(additional_options.value)

        args = " ".join(args_list)
        profile = {"args": args}
        return profile

    @staticmethod
    def _get_targets_from_db(prev_scan_id):
        session = new_session()
        targets = session.query(DebbyScanResults.ip, DebbyScanResults.port)\
                         .filter(DebbyScanResults.scan_id == prev_scan_id)\
                         .filter(DebbyScanResults.enabled == True).distinct()
        target_list = [ip for (ip, port) in targets]
        session.close()

        target_list = list(set(target_list))

        return target_list

    @staticmethod
    def _get_targets_from_project(project_id):
        targets = set()

        s = new_session()
        project = s.query(DebbyProject).filter(DebbyProject.id == project_id).first()
        s.close()

        targets_ = project.targets.split(', ')

        for t in targets_:
            splitted = t.strip().rsplit(':', 1)

            target = splitted[0]

            if target[0] == '[' and target[-1] == ']':
                target = target[1:-1]

            for ip in target_generator_helper([target]):
                targets.add(ip)

        return list(targets)

    @staticmethod
    def new_tasks_payloads_generator(project_id, scan_id=None):
        s = new_session()
        project = s.query(DebbyProject).filter(DebbyProject.id == project_id).first()
        policy = s.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).first()
        s.close()

        targets = list()
        if scan_id:
            session = new_session()
            scan_pipeline = session.query(PipelineScans).filter(PipelineScans.next_scan_id == scan_id).first()
            session.close()
            if scan_pipeline:
                targets = ZookeeperCheckerEngine._get_targets_from_db(scan_pipeline.prev_scan_id)
            else:
                targets = None

        if targets is None:
            targets = ZookeeperCheckerEngine._get_targets_from_project(project_id)

        total = len(targets)
        cnt = 0
        while cnt < total:

            some_targets = targets[cnt:cnt+256]

            payload = json.dumps({
                "engine": project.engine,
                "profile": ZookeeperCheckerEngine._prepare_profile(policy, some_targets),
                "save_to_db": project.save_to_db
            })

            yield (some_targets, payload)

            cnt += 256

        return

    @staticmethod
    def scan_results_to_splunk_events(scan_results, task_id, only_enabled=False):
        """

        :param results:
        :param task_id:
        :param scan:
        :param project:
        :param policy:
        :return:
        """

        events = list()
        cached_resolver = CachedRespResolver()

        session = new_session()
        task = session.query(DebbyTask).filter(DebbyTask.id == task_id).first()
        scan = session.query(DebbyScan).filter(DebbyScan.id == task.debbyscan_id).first()
        project = session.query(DebbyProject).filter(DebbyProject.id == scan.project_id).first()
        policy = session.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).first()
        tags = session.query(DebbyTag).filter(RelationProjectTag.project_id == project.id)\
                                      .filter(RelationProjectTag.tag_id == DebbyTag.id).all()
        session.close()

        tag_list = list([tag.value for tag in tags])

        for res in scan_results:

            """
            scan_results = {
                'ip': ip,
                'port': port,
                'auth_required': None,
                'exception': None,
                'command_result': None,
                'unknown_exception': None,
                'enabled': True,
            }
            """

            enabled = res.get('enabled')
            if only_enabled and not enabled:
                continue

            addr = res.get('ip')
            port = res.get('port')
            scripts = {
                'zookeeper_checker_auth_required': res.get('auth_required'),
                'zookeeper_checker_command_result': res.get('command_result'),
                'zookeeper_checker_exception': res.get('exception'),
                'zookeeper_checker_unknown_exception': res.get(
                    'unknown_exception'),

            }

            resp, source = cached_resolver.get_resps(addr)
            resp = ','.join(resp)

            events.append({
                'event_type': 'info',

                'projectId': project.id,
                'projectName': project.name,
                'engine': project.engine,
                'logClosed': project.log_closed,
                'tags': tag_list,

                'policyId': policy.id,

                'dest_ip': addr,
                'resp': resp,
                'resp_source': source,
                'protocol': addr_to_proto(addr),

                'dest_port': port,
                'transport': PROTO_TCP,
                'time': None,
                'enabled': enabled,
                'scripts': scripts,

                'service_name': None,
                'service_product': None,
                'service_version': None,

                'scanId': scan.id,
                'scanStartTime': datetime_msk_2_timestamp_utc(scan.create_time),

                'taskId': task.id,
            })

        return events

