

import socket
import struct


def ip2int4(ipstr):
    # try:
    #     ipstr = ipstr.decode()
    # except AttributeError:
    #     pass
    return struct.unpack('!I', socket.inet_aton(ipstr))[0]


def ip2int6(ipstr):
    # try:
    #     ipstr = ipstr.decode()
    # except AttributeError:
    #     pass
    val1, val2 = struct.unpack(
        '!QQ', socket.inet_pton(socket.AF_INET6, ipstr),
    )
    return (val1 << 64) + val2


def ip2int(ipstr):
    """Converts the classical decimal, dot-separated, string
    representation of an IP address to an integer, suitable for
    database storage.

    """
    # try:
    #     ipstr = ipstr.decode()
    # except AttributeError:
    #     pass
    try:
        return struct.unpack('!I', socket.inet_aton(ipstr))[0], 'ipv4'
    except socket.error:
        val1, val2 = struct.unpack(
            '!QQ', socket.inet_pton(socket.AF_INET6, ipstr),
        )
        return (val1 << 64) + val2, 'ipv6'


def int2ip4(ipint):
    return socket.inet_ntoa(struct.pack('!I', ipint))


def int2ip6(ipint):
    return socket.inet_ntop(
        socket.AF_INET6,
        struct.pack('!QQ', ipint >> 64, ipint & 0xffffffffffffffff),
    )


def int2ip(ipint):
    """Converts the integer representation of an IP address to its
    classical decimal, dot-separated, string representation.

    """
    try:
        if ipint > 0xffffffff:  # Python 2.6 would handle the overflow
            raise struct.error()
        return socket.inet_ntoa(struct.pack('!I', ipint))
    except struct.error:
        return socket.inet_ntop(
            socket.AF_INET6,
            struct.pack('!QQ', ipint >> 64, ipint & 0xffffffffffffffff),
        )


def int2mask(mask, version='ipv4'):
    """Converts the number of bits set to 1 in a mask (the 24 in
    10.0.0.0/24) to the integer corresponding to the IP address of the
    mask (ip2int("255.255.255.0") for 24)
    IPv6 SUPPORTED TOO

    From scapy:utils.py:itom(x).

    """
    if version == 'ipv4':
        return (0xffffffff00000000 >> mask) & 0xffffffff
    elif version == 'ipv6':
        return (0xffffffffffffffffffffffffffffffff00000000000000000000000000000000 >> mask) & 0xffffffffffffffffffffffffffffffff


def net2range(network):
    """Converts a network to a (start, stop) tuple."""
    try:
        network = network.decode()
    except AttributeError:
        pass
    addr, mask = network.split('/')
    addr, version = ip2int(addr)
    mask = int2mask(int(mask), version)
    start = addr & mask
    if version == 'ipv4':
        stop = int2ip4(start + 0xffffffff - mask)
        start = int2ip4(start)
    else:
        stop = int2ip6(start + 0xffffffffffffffffffffffffffffffff - mask)
        start = int2ip6(start)
    return start, stop

def range2nets(rng, proto='ipv4'):
    """Converts a (start, stop) tuple to a list of networks."""
    start, stop = rng
    if isinstance(start, str):
        start, proto = ip2int(start)
    if isinstance(stop, str):
        stop, proto_stop = ip2int(stop)
    if stop < start or proto != proto_stop:
        raise ValueError()
    res = []
    cur = start
    maskint = 32
    if proto != 'ipv4':
        maskint = 128
    mask = int2mask(maskint, proto)

    f_mask = 0xffffffff
    if proto != 'ipv4':
        f_mask = 0xffffffffffffffffffffffffffffffff

    while True:
        while cur & mask == cur and cur | (~mask & f_mask) <= stop:
            maskint -= 1
            mask = int2mask(maskint, proto)
        res.append('%s/%d' % (int2ip(cur), maskint + 1))
        mask = int2mask(maskint + 1, proto)
        if stop & mask == cur:
            return res
        cur = (cur | (~mask & f_mask)) + 1
        maskint = 32
        if proto != 'ipv4':
            maskint = 128
        mask = int2mask(maskint, proto)
