

import requests
import traceback
import tenacity
import ipaddress

from app.tvm import get_debby_targets_tvm_ticket
from app.db.models import Macro
from app.db.db import new_session
from app.settings import DEFAULT_YANDEXNETS6
from app.utils import str_to_ipnetwork6, str_to_ipaddress6, str_to_ipnetwork6
from .utils import fetch_dnscache_targets
from app.settings import FEATURE_ENABLE_DEBBY_TARGETS


class MacroBaseObject(object):
    def __init__(self, value=None):
        self._value = value.strip() if value else value
        self._targets = None
        self._hardcoded = {
            '_DEBBY_ALL_IPV6_': self._debby_all_ipv6_,
            '_DEBBY_DNSCACHE_IPV6_': self._debby_dnscache_ipv6_,
            '_DEBBY_NETMAP_IPV6_': self._debby_netmap_ipv6_,
            '_DEBBY_CLOUD_NETBOX_': self._debby_cloud_netbox_,
            '_DEBBY_CLOUD_FQDNS_': self._debby_cloud_fqdns_,
            '_DEBBY_TEST_IPV4_': self._debby_test_ipv4_,
            # '_DEBBY_TEST_DNS_': self._debby_test_dns_,
        }

    def get_hardcoded_keys(self):
        return list(self._hardcoded.keys())

    @staticmethod
    def _try_get_macro(macro_name):
        session = new_session()
        macro = session.query(Macro).filter(Macro.value == macro_name).first()
        session.close()

        m_value = []
        if macro:
            m_value = macro.targets.split(', ')

        return m_value

    @staticmethod
    def _debby_cloud_fqdns_imp_():
        URL = "https://c.yandex-team.ru/api/groups2hosts/cloud_prod"

        try:
            res = requests.get(URL)
            fqdns = res.text.split('\n')[:-1]
            return fqdns

        # TODO: Concrete exception
        except Exception:
            traceback.print_exc()
            return []

    def _debby_cloud_fqdns_(self):
        self._targets = self._debby_cloud_fqdns_imp_()
        return self._targets

    @staticmethod
    def _debby_cloud_netbox_imp_():
        def get_targets_by_role(role_):
            base_url = 'https://netbox.cloud.yandex.net/netbox-public/api/ipam/prefixes/?role='
            url = base_url + role_
            targets_ = []

            while url:
                r = requests.get(url, verify="/etc/ssl/certs")
                data = r.json()
                for result in data['results']:
                    targets_.append(result['prefix'])
                url = data['next']

            return targets_

        try:
            roles = ['loopbacks', 'public-technical-nets']
            targets = []

            for role in roles:
                targets += get_targets_by_role(role)

            ftargets = list()
            for target in targets:
                try:
                    net = ipaddress.IPv4Network(target)
                    # net = ipaddress.IPv4Network(target.decode())
                    if net.prefixlen >= 24:
                        ftargets.append(target)
                except ipaddress.AddressValueError:
                    pass

            return ftargets

        # TODO: Concrete exception
        except Exception:
            traceback.print_exc()
            return []

    def _debby_cloud_netbox_(self):
        self._targets = self._debby_cloud_netbox_imp_()
        return self._targets

    def _debby_dnscache_ipv6_(self):
        self._targets = fetch_dnscache_targets()
        return self._targets

    def _debby_all_ipv6_(self):
        targets1 = fetch_dnscache_targets()
        targets2 = self._debby_netmap_ipv6_()
        self._targets = list(set(targets1+targets2))
        return self._targets

    def _debby_netmap_ipv6_(self):
        # Make request to rocktables
        try:
            r = requests.get('https://ro.racktables.yandex.net/export/netmap/L23.v46')
        except requests.ConnectionError:
            print('[+] fetch_ipv6_addresses. Unable to connect to https://ro.racktables.yandex.net/export/netmap/L23.v46...')
            self._targets = []
            return self._targets

        # get _YANDEXNETS6_ ipaddress.IPv6Network objects
        yandex_nets_6 = self._try_get_macro('_YANDEXNETS6_')
        if not yandex_nets_6:
            yandex_nets_6 = DEFAULT_YANDEXNETS6[:]
        yandex_nets_6 = list([str_to_ipnetwork6(net) for net in yandex_nets_6])

        # Parse results and build ipv6_list
        ipv6_list = []
        s = r.text
        left_idx = 0
        while True:
            idx = s.find('\n', left_idx)
            ss = s[left_idx:idx]

            if len(ss) == 0 or idx == -1:
                break

            addr_str = ss.split('\t')[0]
            addr_obj = str_to_ipaddress6(addr_str)

            if addr_obj:
                is_addr_in_yandex_nets_6_list = list([addr_obj in net for net in yandex_nets_6])
                is_addr_in_yandex_nets_6 = any(is_addr_in_yandex_nets_6_list)

                if addr_obj and addr_obj.is_global and is_addr_in_yandex_nets_6:
                    ipv6_list.append(addr_str)

            left_idx = idx+1

        self._targets = ipv6_list
        return self._targets

    def _debby_test_ipv4_(self):
        self._targets = ['5.255.255.5']
        return self._targets
    #
    # def _debby_test_dns_(self):
    #     self._targets = ['yandex.ru']
    #     return self._targets

    def _yandex_macro_(self):
        self._targets = self._try_get_macro(self._value)
        return self._targets

    def _smart_macro_(self):
        ipv6nets = list()   # list of nets in original macro
        others = list()     # list of elements in original macros which are not ipv6nets
        ipv6_selected = list() # list of ipvt targets from _DEBBY_ALL_IPV6_ inside ipv6nets

        # split ipv6 network and other targets
        raw_targets = self._try_get_macro(self._value[len("SMART"):])
        for raw_target in raw_targets:
            ipv6net = str_to_ipnetwork6(raw_target)
            if ipv6net:
                ipv6nets.append(ipv6net)
            else:
                others.append(raw_target)

        # fetch DEBBY_ALL_IPV6 targets
        ipv6_raw_targets = self._debby_all_ipv6_()

        # check elems from DEBBY_ALL_IPV6 if they are in ipv6nets from original macro
        for ipv6_raw_target in ipv6_raw_targets:

            ipv6_target = str_to_ipaddress6(ipv6_raw_target)
            if not ipv6_target:
                continue

            for ipv6net in ipv6nets:
                if ipv6_target in ipv6net:
                    ipv6_selected.append(ipv6_raw_target)

        # concatenate selected ipv6 and other elements
        self._targets = ipv6_selected + others
        return self._targets

    def _fetch_targets_from_debby_targets(self, macro):
        tvm_ticket = get_debby_targets_tvm_ticket()
        r = requests.get("https://debby-targets.sec.yandex-team.ru/macros/{}".format(macro), headers={'X-Ya-Service-Ticket': tvm_ticket}, verify="/etc/ssl/certs")
        return r.text.split(",") if len(r.text) != 0 else None

    @tenacity.retry(retry=tenacity.retry_if_result(lambda res: not res),
                    wait=tenacity.wait_fixed(1), stop=tenacity.stop_after_attempt(3))
    def _get_targets_imp(self):
        """ Return list of targets (IPs or domain names) """

        if not self._targets and FEATURE_ENABLE_DEBBY_TARGETS:
            self._targets = self._fetch_targets_from_debby_targets(self._value)

        if self._targets:
            return self._targets

        try:
            return self._hardcoded[self._value]()
        except KeyError:
            if self._value.startswith("SMART_"):
                return self._smart_macro_()
            else:
                return self._yandex_macro_()

    def get_targets(self):
        """ Return list of targets (IPs or domain names) """

        try:
            targets = self._get_targets_imp()
            return targets
            # fix macros with symbol "@"
            # targets = list(map(lambda x:x[x.find("@")+1:], targets))
            # return list(set(targets))
        except tenacity.RetryError:
            return list()
