

import sys
from flask import Blueprint, render_template, request, redirect, url_for, g

from app.db.db import new_session
from app.db.models import DebbyPolicy, DebbyPolicyScript, DebbyPolicyAdditionalOptions
from app import yauth
from app.validators import DebbyValidateException, validate_new_policy
from app.settings import TRANSPORT_PROTOCOLS, AVAILABLE_NMAP_SCRIPTS
from app.policies.utils import create_policy, edit_policy

bp = Blueprint('policies', __name__, url_prefix='/policies')


@bp.route('/', methods=['GET'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def policies_main():
    session = new_session()
    policies = session.query(DebbyPolicy).all()
    for policy in policies:
        scripts_objs = session.query(DebbyPolicyScript).filter(DebbyPolicyScript.policy_id == policy.id).all()
        scripts_str = ", ".join(scripts_obj.name for scripts_obj in scripts_objs)
        policy.str_scripts = scripts_str

        additional_options = session.query(DebbyPolicyAdditionalOptions)\
                                    .filter(DebbyPolicyAdditionalOptions.policy_id == policy.id).first()
        policy.additional_options = additional_options.value if additional_options else ''
    session.close()
    return render_template('policies_main.html', policies=policies)


@bp.route('/new', methods=['GET', 'POST'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def policies_new():

    if request.method == 'GET':
        return render_template('policies_new.html', transport_protocols=TRANSPORT_PROTOCOLS,
                               available_nmap_scripts=AVAILABLE_NMAP_SCRIPTS)

    else:  # POST
        try:
            VARS = validate_new_policy(request)

            create_policy(VARS['name'], VARS['scan_type'], VARS['ports_list'],
                          VARS['spec_scripts'], VARS['additional_options'])

            return redirect(url_for('policies.policies_main'))

        except DebbyValidateException as dex:
            return render_template('policies_new.html', error=dex.args, transport_protocols=TRANSPORT_PROTOCOLS,
                                   available_nmap_scripts=AVAILABLE_NMAP_SCRIPTS)


@bp.route('/<int:policy_id>', methods=['DELETE'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def policy_delete(policy_id):

    session = new_session()
    session.query(DebbyPolicy).filter(DebbyPolicy.id == policy_id).delete()
    session.commit()
    session.close()

    return redirect(url_for('policies.policies_main'))


@bp.route('/<int:policy_id>', methods=['GET', 'POST'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def policy_edit(policy_id):
    session = new_session()
    policy = session.query(DebbyPolicy).filter(DebbyPolicy.id == policy_id).first()
    scripts_objs = session.query(DebbyPolicyScript).filter(DebbyPolicyScript.policy_id == policy.id).all()
    policy.scripts_names = [script_obj.name for script_obj in scripts_objs]

    additional_options = session.query(DebbyPolicyAdditionalOptions) \
        .filter(DebbyPolicyAdditionalOptions.policy_id == policy.id).first()
    policy.additional_options = additional_options.value if additional_options else ''
    session.close()

    if request.method == 'GET':
        return render_template('policies_edit.html', policy=policy, transport_protocols=TRANSPORT_PROTOCOLS,
                               available_nmap_scripts=AVAILABLE_NMAP_SCRIPTS)
    else:
        try:
            VARS = validate_new_policy(request)

            edit_policy(policy_id, VARS['name'], VARS['scan_type'], VARS['ports_list'],
                        VARS['spec_scripts'], VARS['additional_options'])

            return redirect(url_for('policies.policy_edit', policy_id=policy.id))

        except DebbyValidateException as dex:
            return render_template('policies_edit.html', error=dex.args, policy=policy,
                                   transport_protocols=TRANSPORT_PROTOCOLS,
                                   available_nmap_scripts=AVAILABLE_NMAP_SCRIPTS)
