

from flask import Blueprint, render_template, request, redirect, url_for
from builtins import map

from app.utils import datetime_msk_2_timestamp_utc
from app.utils import now
from app.db.db import new_session
from app.db.models import DebbyProject, DebbyPolicy, DebbyScan, PipelineProjects
from app.db.models import RelationProjectTag, DebbyTag, Namespace
from app.settings import ENGINES
from app import yauth
from app.validators import DebbyValidateException, DebbyRuntimeException, validate_project
from app.projects.utils import get_projects, get_project_details, make_project_drawable, create_new_project
from app.tasks import run_scan_for_project


bp = Blueprint('projects', __name__, url_prefix='/projects')


@bp.route('/')
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def projects_main():

    projects = get_projects()

    # convert date to user friendly strings
    for project in projects:
        make_project_drawable(project)

    return render_template('projects_main.html', projects=projects)


@bp.route('/<int:project_id>', methods=['GET'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def projects_details(project_id):

    project, scans = get_project_details(project_id)

    make_project_drawable(project)

    return render_template('projects_details.html', project=project, scans=scans)


@bp.route('/new', methods=['GET', 'POST'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def projects_new():

    cur_time_str = now().strftime('%d.%m.%Y %H:%M')
    session = new_session()
    known_policies = session.query(DebbyPolicy.id, DebbyPolicy.name).all()
    known_tags = session.query(DebbyTag).all()
    known_projects = session.query(DebbyProject).all()
    known_namespaces = [Namespace(id=0, name="Default (TRY TO NOT USE)", contacts="")] + session.query(Namespace).order_by(Namespace.id).all()
    session.close()

    if request.method == 'GET':
        return render_template('projects_new.html', cur_time=cur_time_str, projects=known_projects,
                               policies=known_policies, engines=ENGINES, tags=known_tags, namespaces=known_namespaces)

    else:
        try:
            # validate params
            VARS = validate_project(request, known_tags, known_policies, known_projects, known_namespaces)
            VARS['save_to_db'] = False
            if not VARS["contacts"] and "yalogin" in g:
                VARS["contacts"] = g.yalogin

            # create project
            create_new_project(VARS['name'], VARS['target_list'], VARS['engine'], VARS['policy_id'],
                               VARS['dt_start'], VARS['dt_stop'], VARS['ts_interval'],
                               VARS['spec_tags_ids'], VARS['spec_pipeline_projs_ids'],
                               max_scan_time=VARS['max_scan_time'], st_ticket=VARS['st_ticket'],
                               retries=VARS['retries'], contacts=VARS['contacts'], retry_period=VARS['retry_period'],
                               log_closed=VARS['log_closed'], save_to_db=VARS['save_to_db'], namespace_id=VARS['namespace_id'])

            # redirect to project's main page
            return redirect(url_for('projects.projects_main'))

        # handle validation exception
        except DebbyValidateException as ex:
            return render_template('projects_new.html', cur_time=cur_time_str, engines=ENGINES, projects=known_projects,
                                   error='DebbyValidateException: ' + repr(ex),
                                   policies=known_policies, tags=known_tags, namespaces=known_namespaces)

        # handle runtime exception
        except DebbyRuntimeException as ex:
            return render_template('projects_new.html', cur_time=cur_time_str, engines=ENGINES, projects=known_projects,
                                   error='DebbyRuntimeException: ' + repr(ex),
                                   policies=known_policies, tags=known_tags, namespaces=known_namespaces)


@bp.route('/<int:project_id>/edit', methods=['GET', 'POST'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def project_edit(project_id):

    # Get info from db
    session = new_session()
    project = session.query(DebbyProject).filter(DebbyProject.id == project_id).first()
    known_policies = session.query(DebbyPolicy.id, DebbyPolicy.name).all()
    known_tags = session.query(DebbyTag).all()
    currently_selected_tags = session.query(DebbyTag).filter(RelationProjectTag.tag_id == DebbyTag.id) \
                                                     .filter(RelationProjectTag.project_id == project.id).all()
    currently_selected_pipelines = session.query(PipelineProjects)\
                                          .filter(PipelineProjects.next_proj_id == project_id).all()
    known_projects = session.query(DebbyProject).all()
    known_namespaces = [Namespace(id=0, name="Default (TRY TO NOT USE)", contacts="")] + session.query(Namespace).all()
    session.close()

    if request.method == 'GET':

        # prepare data
        project.str_scan_start = project.scan_start.strftime("%d.%m.%Y %H:%M")
        project.str_scan_stop = project.scan_stop.strftime("%d.%m.%Y %H:%M")
        project.int_interval = datetime_msk_2_timestamp_utc(project.interval)

        for tag in known_tags:
            if tag in currently_selected_tags:
                tag.is_selected = True

        currently_selected_pipelines_prev_proj_ids = list([pp.prev_proj_id for pp in currently_selected_pipelines])
        for proj in known_projects:
            if proj.id in currently_selected_pipelines_prev_proj_ids:
                proj.is_selected = True

        return render_template('project_edit.html', project=project, engines=ENGINES, projects=known_projects,
                               policies=known_policies, tags=known_tags, namespaces=known_namespaces)

    else:
        try:
            VARS = validate_project(request, known_tags, known_policies, known_projects, known_namespaces)
            VARS['save_to_db'] = False

            updating = {DebbyProject.name: VARS['name'],
                        DebbyProject.targets: VARS['target_list'],
                        DebbyProject.engine: VARS['engine'],
                        DebbyProject.policy_id: VARS['policy_id'],
                        DebbyProject.scan_start: VARS['dt_start'],
                        DebbyProject.scan_stop: VARS['dt_stop'],
                        DebbyProject.interval: VARS['ts_interval'],
                        DebbyProject.log_closed: VARS['log_closed'],
                        DebbyProject.save_to_db: VARS['save_to_db'],
                        DebbyProject.st_ticket: VARS['st_ticket'],
                        DebbyProject.retries: VARS['retries'],
                        DebbyProject.retry_period: VARS['retry_period'],
                        DebbyProject.contacts: VARS['contacts'],
                        DebbyProject.namespace_id: VARS['namespace_id'],
                        DebbyProject.max_scan_time: VARS['max_scan_time']}

            session = new_session()
            session.query(DebbyProject).filter(DebbyProject.id == project.id).update(updating)
            session.query(RelationProjectTag).filter(RelationProjectTag.project_id == project.id).delete()
            session.query(PipelineProjects).filter(PipelineProjects.next_proj_id == project.id).delete()

            # Add new tags
            for tag_id in VARS['spec_tags_ids']:
                ret = RelationProjectTag(project_id=project.id, tag_id=tag_id)
                session.add(ret)

            # Add pipeline projects info
            for pline_proj_id in VARS['spec_pipeline_projs_ids']:
                pp = PipelineProjects(prev_proj_id=pline_proj_id, next_proj_id=project.id)
                session.add(pp)

            session.commit()
            session.close()

            return redirect(url_for('projects.projects_main'))

        except DebbyValidateException as ex:
            return render_template('project_edit.html', project=project, engines=ENGINES, projects=known_projects,
                                   error=ex.args, policies=known_policies, tags=known_tags)


@bp.route('/<int:project_id>/fork', methods=['GET', 'POST'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def project_fork(project_id):

    # Get info from db
    session = new_session()
    project = session.query(DebbyProject).filter(DebbyProject.id == project_id).first()
    known_policies = session.query(DebbyPolicy.id, DebbyPolicy.name).all()
    known_tags = session.query(DebbyTag).all()
    currently_selected_tags = session.query(DebbyTag).filter(RelationProjectTag.tag_id == DebbyTag.id) \
                                                     .filter(RelationProjectTag.project_id == project.id).all()
    currently_selected_pipelines = session.query(PipelineProjects)\
                                          .filter(PipelineProjects.next_proj_id == project_id).all()
    known_projects = session.query(DebbyProject).all()
    known_namespaces = [Namespace(id=0, name="Default (TRY TO NOT USE)", contacts="")] + session.query(Namespace).all()
    session.close()

    if request.method == 'GET':

        # prepare data
        project.str_scan_start = project.scan_start.strftime("%d.%m.%Y %H:%M")
        project.str_scan_stop = project.scan_stop.strftime("%d.%m.%Y %H:%M")
        project.int_interval = datetime_msk_2_timestamp_utc(project.interval)

        for tag in known_tags:
            if tag in currently_selected_tags:
                tag.is_selected = True

        currently_selected_pipelines_prev_proj_ids = list([pp.prev_proj_id for pp in currently_selected_pipelines])
        for proj in known_projects:
            if proj.id in currently_selected_pipelines_prev_proj_ids:
                proj.is_selected = True

        return render_template('project_fork.html', project=project, engines=ENGINES, projects=known_projects,
                               policies=known_policies, tags=known_tags, namespaces=known_namespaces)

    else:
        try:
            VARS = validate_project(request, known_tags, known_policies, known_projects)
            VARS['save_to_db'] = False

            # Create scan
            de = DebbyProject(name=VARS['name'], targets=VARS['target_list'], engine=VARS['engine'], 
                              policy_id=VARS['policy_id'], scan_start=VARS['dt_start'], scan_stop=VARS['dt_stop'], 
                              interval=VARS['ts_interval'], max_scan_time=VARS['max_scan_time'], 
                              log_closed=VARS['log_closed'], save_to_db=VARS['save_to_db'], st_ticket=VARS['st_ticket'],
                              retries=VARS['retries'], contacts=VARS['contacts'],
                              retry_period=VARS['retry_period'], namespace_id=VARS['namespace_id'])
            # Add to db
            session = new_session()
            session.add(de)
            session.commit()

            # Relation between project and required tags
            for tag_id in VARS['spec_tags_ids']:
                ret = RelationProjectTag(project_id=de.id, tag_id=tag_id)
                session.add(ret)

            # Add pipeline projects info
            for pline_proj_id in VARS['spec_pipeline_projs_ids']:
                pp = PipelineProjects(prev_proj_id=pline_proj_id, next_proj_id=de.id)
                session.add(pp)

            session.commit()
            session.close()

            # redirect to project's main page
            return redirect(url_for('projects.projects_main'))

        except DebbyValidateException as ex:
            return render_template('project_fork.html', project=project, engines=ENGINES, projects=known_projects,
                                   error=ex.args, policies=known_policies, tags=known_tags, namespaces=known_namespaces)


@bp.route('/<int:project_id>/pause', methods=['PUSH'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def project_pause(project_id):

    session = new_session()
    session.query(DebbyProject).filter(DebbyProject.id == project_id).update({DebbyProject.paused: True})
    session.commit()
    session.close()

    return redirect(url_for('projects.projects_main'))


@bp.route('/<int:project_id>/continue', methods=['PUSH'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def project_continue(project_id):

    session = new_session()
    session.query(DebbyProject).filter(DebbyProject.id == project_id).update({DebbyProject.paused: False})
    session.commit()
    session.close()

    return redirect(url_for('projects.projects_main'))


@bp.route('/<int:project_id>', methods=['DELETE'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def project_delete(project_id):

    session = new_session()
    session.query(DebbyProject).filter(DebbyProject.id == project_id).update({
                                                                        DebbyProject.paused: True,
                                                                        DebbyProject.deleted: True})
    session.commit()
    session.close()

    return redirect(url_for('projects.projects_main'))


@bp.route('/<int:project_id>/run_scan_now', methods=['POST'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def run_scan_now(project_id):

    run_scan_for_project.delay(project_id)
    return redirect(url_for('projects.projects_main'))
