

from builtins import map

import traceback

from app.settings import ENGINE_NMAP, API_PROJECTS_LIMIT
from app.validators import DebbyRuntimeException, DebbyOwnerException, DebbyValidateException
from app.db.db import new_session
from app.db.models import DebbyProject, DebbyPolicy, DebbyTag, RelationProjectTag, PipelineProjects, DebbyScan
from app.db.models import ProjectApiInfo
from app.utils import now, datetime_msk_2_timestamp_utc, eprint


def get_projects(tag_info=True, pipeline_info=True, namespace_id=None):

    # start db session
    session = new_session()

    # get (all) (not deleted) projects in desc order
    q = session.query(DebbyProject, DebbyPolicy).filter(DebbyProject.deleted == False)
    q = q.filter(DebbyProject.policy_id == DebbyPolicy.id)
    if namespace_id:
        q = q.filter(DebbyProject.namespace_id == namespace_id)
    elif namespace_id == 0:
        q = q.filter(DebbyProject.namespace_id == None)
    q = q.order_by(DebbyProject.id.desc())
    projects = q.all()

    # adding tags and pipelines info
    if tag_info or pipeline_info:
        for project in projects:
            # add tags
            if tag_info:
                # get all tags for specific project
                project_tags = session.query(DebbyTag).filter(DebbyTag.id == RelationProjectTag.tag_id)\
                                      .filter(RelationProjectTag.project_id == project.DebbyProject.id).all()
                # list to string
                project.DebbyProject.tags_ = project_tags

            # add pipelines
            if pipeline_info:
                # get all pipelines for specific project

                pipeline_projects = session.query(DebbyProject)\
                                           .filter(DebbyProject.id == PipelineProjects.prev_proj_id) \
                                           .filter(PipelineProjects.next_proj_id == project.DebbyProject.id) \
                                           .all()
                project.DebbyProject.pipelines_ = pipeline_projects

    session.close()

    return projects


def get_project_details(project_id):

    # new db session
    session = new_session()

    # get scans for project
    scans = session.query(DebbyScan).filter(DebbyScan.project_id == project_id).order_by(DebbyScan.id.desc()).all()

    # get project info
    project = session.query(DebbyProject, DebbyPolicy).filter(DebbyProject.policy_id == DebbyPolicy.id)\
                                                      .filter(DebbyProject.id == project_id)\
                                                      .first()

    # get tags for project
    tags = session.query(DebbyTag.value).filter(RelationProjectTag.tag_id == DebbyTag.id)\
                                        .filter(RelationProjectTag.project_id == project.DebbyProject.id)\
                                        .all()
    project.DebbyProject.tags_ = tags

    # get pipeline info
    pipeline_projects = session.query(DebbyProject).filter(DebbyProject.id == PipelineProjects.prev_proj_id) \
                                                   .filter(PipelineProjects.next_proj_id == project.DebbyProject.id)\
                                                   .all()
    project.DebbyProject.pipelines_ = pipeline_projects

    # close db session
    session.close()

    return project, scans


def make_project_drawable(project, bool_is_finished=True, int_interval=True, tags_=True, pipelines_=True):
    # set: is project finished
    if bool_is_finished:
        project.DebbyProject.bool_is_finished = now() > project.DebbyProject.scan_stop

    # set: convert datetime to int (seconds)
    if int_interval:
        project.DebbyProject.int_interval = datetime_msk_2_timestamp_utc(project.DebbyProject.interval)

    # set: project tags string
    if tags_:
        project.DebbyProject.tags_ = ', '.join(list([tag_obj.value for tag_obj in project.DebbyProject.tags_]))

    # pipelines
    if pipelines_:
        pp_array = list(['{}:{}'.format(pp.id, pp.name) for pp in project.DebbyProject.pipelines_])
        project.DebbyProject.pipelines_ = ', '.join(pp_array)

    # handle empty st_ticket
    # if st_ticket:
    #     project.DebbyProject.st_ticket_ = project.DebbyProject.st_ticket if project.DebbyProject.st_ticket else ""

    return


def create_new_project(name, targets, engine, policy_id, scan_start, scan_stop,
                       interval, tags_list, project_prev_pipelines_list, 
                       max_scan_time=0, log_closed=False, save_to_db=False,
                       st_ticket=None, retries=0, contacts='', retry_period=0, namespace_id=None):
    """
    Create project

    :param name:
        string
    :param targets:
        string: "target1, target2, ..."
    :param engine:
        string
    :param policy_id:
        integer
    :param scan_start:
        datetime
    :param scan_stop:
        datetime
    :param interval:
        datetime
    :param tags_list:
        list of integers
    :param project_prev_pipelines_list:
        list of integers
    :param log_closed:
        boolean
    :param save_to_db:
        boolean
    :param st_ticket:
        string or None
    :param retries:
        int
    :param retry_period:
        int
    :param contacts:
        string
    :param namespace_id:
        int or None

    :return:
        newly created project id

    :raises:
        DebbyRuntimeException
    """

    try:
        # Create scan
        new_project = DebbyProject(name=name, targets=targets, engine=engine, policy_id=policy_id,
                                   scan_start=scan_start, scan_stop=scan_stop, interval=interval,
                                   log_closed=log_closed, save_to_db=save_to_db, max_scan_time=max_scan_time,
                                   st_ticket=st_ticket, retries=retries, contacts=contacts,
                                   retry_period=retry_period, namespace_id=namespace_id)
        # Add to db
        session = new_session()
        session.add(new_project)
        session.flush()

        new_project_id = new_project.id

        # Relation between project and required tags
        for tag_id in tags_list:
            ret = RelationProjectTag(project_id=new_project_id, tag_id=tag_id)
            session.add(ret)

        # Add pipeline projects info
        for pline_proj_id in project_prev_pipelines_list:
            pp = PipelineProjects(prev_proj_id=pline_proj_id, next_proj_id=new_project_id)
            session.add(pp)

        session.commit()
        session.close()

        return new_project_id

    except Exception as e:
        traceback.print_exc()
        raise DebbyRuntimeException('Saving to database error: {}'.format(e))


def create_projectapiinfo(project_id, api_src_id, is_public):
    s = new_session()
    project_api_info = ProjectApiInfo(project_id=project_id, api_src_id=str(api_src_id), is_public=is_public)
    s.add(project_api_info)
    s.commit()
    s.close()


def get_projects_by_api_src_id(api_src_id):
    s = new_session()
    project_ids = s.query(ProjectApiInfo.project_id).filter(ProjectApiInfo.api_src_id == str(api_src_id))
    projects = s.query(DebbyProject.id, DebbyProject.name).filter(DebbyProject.id.in_(project_ids))\
                                                          .filter(DebbyProject.deleted == False).all()
    projects_list = list([{'id': p.id, 'name': p.name} for p in projects])
    s.close()
    return projects_list


def delete_project_by_id_and_api_src_id(project_id, api_src_id):
    s = new_session()

    project = s.query(DebbyProject).filter(DebbyProject.id == project_id).first()
    if not project:
        s.close()
        raise DebbyRuntimeException('Unknown project identifier')

    pai = s.query(ProjectApiInfo).filter(ProjectApiInfo.project_id == project_id) \
                                 .filter(ProjectApiInfo.api_src_id == str(api_src_id)).first()
    if not pai:
        s.close()
        raise DebbyOwnerException('Permission Denied')

    if pai.is_public == False:
        s.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).delete()

    s.delete(pai)
    s.delete(project)
    s.commit()
    s.close()


def get_project_info_by_id_and_api_src_id(project_id, api_src_id, get_targets=False, get_policy=False):
    s = new_session()

    project = s.query(DebbyProject).filter(DebbyProject.id == project_id).first()
    if not project:
        s.close()
        raise DebbyRuntimeException('Unknown project identifier')

    pai = s.query(ProjectApiInfo).filter(ProjectApiInfo.project_id == project_id)\
                                 .filter(ProjectApiInfo.api_src_id == str(api_src_id)).first()
    if not pai:
        s.close()
        raise DebbyOwnerException('Permission Denied')

    scans = s.query(DebbyScan).filter(DebbyScan.project_id == project_id).all()

    s.close()

    scans_info = list([{
            'id': scan.id,
            'create_time': datetime_msk_2_timestamp_utc(scan.create_time),
            'finish_time': datetime_msk_2_timestamp_utc(scan.finish_time) if scan.finish_time else None,
            'state': scan.state
        } for scan in scans])

    project_info = {
        'scans': scans_info,
        'id': project.id,
        'name': project.name,
        'engine': project.engine,
        'scan_start': datetime_msk_2_timestamp_utc(project.scan_start),
        'scan_stop': datetime_msk_2_timestamp_utc(project.scan_stop) if project.scan_stop else None,
        'interval': datetime_msk_2_timestamp_utc(project.interval) if project.interval else None,
        'max_scan_time': project.max_scan_time,
    }

    if get_targets:
        project_info['targets'] = project.targets

    if get_policy and project.engine == ENGINE_NMAP:
        policy = s.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).first()
        project_info['transport'] = policy.scan_type
        project_info['ports'] = policy.ports

    return project_info


def check_projects_limit_for_api_src_id(api_src_id):
    s = new_session()
    cnt = s.query(ProjectApiInfo).filter(ProjectApiInfo.api_src_id == str(api_src_id)).count()
    s.close()

    if cnt >= API_PROJECTS_LIMIT:
        raise DebbyValidateException('You have {} projects. Delete any to create new one'.format(cnt))


def _get_projects_engine_by_id(project_id):
    s = new_session()
    project = s.query(DebbyProject).filter(DebbyProject.id == project_id).first()
    s.close()
    return project.engine
