

import os
import socket
import requests
import dns.resolver
import dns.reversename

QLOUD_EXT_BASE_URL = "https://qloud-ext.yandex-team.ru"
QLOUD_INT_BASE_URL = "https://qloud.yandex-team.ru"

class QloudHelper:
    """
    Usage example:
        qh = QloudHelper(os.getenv("QLOUD_TOKEN"), QLOUD_EXT_BASE_URL)
        fqdn = "staging.med.yandex.ru"
        if qh.is_qloud_host(fqdn):
            routes = qh.find_routes_for_domain(fqdn)
    """

    def __init__(self, oauth_token, base_url):
        self._oauth_token = oauth_token
        self._base_url = base_url

    def _build_auth_headers(self):
        return {
            "Authorization": "OAuth " + self._oauth_token
        }

    def find_object(self, object_str):
        try:
            url = self._base_url + "/api/find-object"
            params = {"pattern": object_str}
            r = requests.get(url, params=params, headers=self._build_auth_headers())
            if r.status_code != 200:
                return None
            return r.json()
        except Exception as e:
            print("[!] Exception. QloudHelper::_find_object(object_str={}) base_url={}. {}".format(object_str, self._base_url, e))
        return None

    def extract_linkobject_for_domain(self, qloud_object_info):
        domains = qloud_object_info.get("DOMAIN")
        if not domains:
            return None
        
        domain = domains[0]
        
        link_object = domain.get("linkObject")
        if not link_object:
            return None

        return link_object

    def find_env_id_for_domain(self, domain_str):
        qloud_object_info = self.find_object(domain_str)
        if not qloud_object_info:
            return None

        qloud_env_id = self.extract_linkobject_for_domain(qloud_object_info)
        return qloud_env_id

    def get_qloud_env(self, env_id):
        try:
            url = self._base_url + "/api/v1/environment/stable/" + env_id 
            r = requests.get(url, headers=self._build_auth_headers())
            if r.status_code != 200:
                return None
            return r.json()
        except Exception as e:
            print("[!] Exception. QloudHelper::get_qloud_env(env_id={}) base_url={}. {}".format(env_id, self._base_url, e))
        return None

    def get_routes_from_qloud_env_data(self, qloud_env_data):
        routes = qloud_env_data.get("routes")
        if not routes:
            return

        locations = [r.get("location") for r in routes]
        return locations

    def find_routes_for_env_id(self, env_id):
        qloud_env_data = self.get_qloud_env(env_id)
        if not qloud_env_data:
            return None

        routes = self.get_routes_from_qloud_env_data(qloud_env_data)
        return routes

    def find_routes_for_domain(self, domain):
        env_id = self.find_env_id_for_domain(domain)
        if not env_id:
            return None

        routes = self.find_routes_for_env_id(env_id)
        return routes

    def get_ptr(self, ip):
        """
        77.88.21.225 -> med-test.stable.qloud-b.yandex.net
        2a02:6b8::1:225 -> med-test.stable.qloud-b.yandex.net
        """
        try:
            addr = dns.reversename.from_address(ip)
            dns_ans = dns.resolver.query(addr, 'PTR')
            return dns_ans[0].to_text().strip('.')
        except dns.resolver.NoAnswer:
            pass
        except Exception as e:
            pass
        return None

    def is_qloud_ip(self, ip):
        ptr = self.get_ptr(ip)
        if ptr and ptr.endswith("qloud-b.yandex.net"):
            return True
        else:
            return False

    def is_qloud_host(self, host):
        fqdn = socket.getfqdn(host)
        return fqdn.endswith("qloud-b.yandex.net")
