

import requests
import tenacity

from app.settings import CRT_TOKEN, CRT_API_URL
from app.target_resolver import TargetResolver
from app.utils import is_valid_ipv4_address, is_valid_ipv6_address


NO_RESPS = list()
NO_SOURCE = None

# # DEPRICATED
# SOURCE_GOLEM = 'golem'
# @tenacity.retry(retry=tenacity.retry_if_exception_type(requests.exceptions.ConnectionError),
#                 wait=tenacity.wait_fixed(2), stop=tenacity.stop_after_attempt(10), reraise=True)
# def _DEPRICATED_get_resps_for_fqdn_from_golem(fqdn):
#     response = requests.get('http://ro.admin.yandex-team.ru/api/get_host_resp.sbml', params={'hostname': fqdn})
#     if response.status_code != 200 or not response.text:
#         return list()

#     resps = list(filter(lambda x: x, map(lambda x: x.strip(), filter(lambda x: x, response.text.split(',')))))
#     return resps


# @tenacity.retry(retry=tenacity.retry_if_exception_type(requests.exceptions.ConnectionError),
#                 wait=tenacity.wait_fixed(2), stop=tenacity.stop_after_attempt(10), reraise=True)
# def _get_resps_for_fqdn_from_crt(fqdn):
#
#     if not CRT_TOKEN:
#         return NO_RESPS
#
#     headers = {'Authorization': 'OAuth {}'.format(CRT_TOKEN)}
#     params = {'_fields': 'ca_name,common_name,abc_service,status', 'common_name': fqdn}
#     response = requests.get(CRT_API_URL, headers=headers, params=params)
#     res = response.json()
#
#     if not res.get('results'):
#         return NO_RESPS
#
#     print(res.get('results', []))
#
#     for crt in res.get('results', []):
#         return crt


class AppSecDiscoveryClient(object):

    @staticmethod
    @tenacity.retry(wait=tenacity.wait_random_exponential(max=10), stop=tenacity.stop_after_attempt(3))
    def _get_resp(json_data):
        url = 'https://appsec-discovery.sec.yandex-team.ru/api/for_services/search_host_owner'
        resp = requests.post(url, json=json_data, verify=False, timeout=10).json()
        results = resp.get('results')
        if results:
            return results.get('owners'), results.get('source')
        else:
            return NO_RESPS, NO_SOURCE

    @staticmethod
    def get_resp_by_ip(ip):
        return AppSecDiscoveryClient._get_resp({'ip': ip})

    @staticmethod
    def get_resp_by_fqdn(fqdn):
        return AppSecDiscoveryClient._get_resp({'fqdn': fqdn})


class RespResolver(object):

    @staticmethod
    def _get_resps_for_ip(ip):
        return AppSecDiscoveryClient.get_resp_by_ip(ip)

    @staticmethod
    def _get_resps_for_fqdn(fqdn):
        # resps = _get_resps_for_fqdn_from_golem(fqdn)
        # if resps:
        #     return resps, SOURCE_GOLEM

        resps, reason = AppSecDiscoveryClient.get_resp_by_fqdn(fqdn)
        if resps:
            return resps, reason

        return NO_RESPS, NO_SOURCE

    @staticmethod
    def get_resps_by_fqdn(fqdn):
        resps, reason = RespResolver._get_resps_for_fqdn(fqdn)
        if resps:
            return resps, reason

        ip_list = TargetResolver.resolve_fqdn(fqdn)
        # try only fqdns
        for ip in ip_list:
            resps, reason = RespResolver.get_resps_by_ip(ip, resolve_ip=False)
            if resps:
                return resps, reason
        # now try ips
        for ip in ip_list:
            resps, reason = RespResolver.get_resps_by_ip(ip, resolve_fqdn=False)
            if resps:
                return resps, reason

        return NO_RESPS, NO_SOURCE

    @staticmethod
    def get_resps_by_ip(ip, resolve_ip=True, resolve_fqdn=True):
        if resolve_fqdn:
            fqdn = TargetResolver.ip_to_fqdn(ip)
            resps, reason = RespResolver._get_resps_for_fqdn(fqdn)
            if resps:
                return resps, reason

        if resolve_ip:
            resps, reason = RespResolver._get_resps_for_ip(ip)
            if resps:
                return resps, reason

        return NO_RESPS, NO_SOURCE

    @staticmethod
    def get_resps(host):
        if not host:
            return NO_RESPS, NO_SOURCE

        if is_valid_ipv4_address(host) or is_valid_ipv6_address(host):
            return RespResolver.get_resps_by_ip(host)

        if '.' in host:
            return RespResolver.get_resps_by_fqdn(host)

        return NO_RESPS, NO_SOURCE


class CachedRespResolver(object):
    def __init__(self):
        self._cache = dict()

    def get_resps(self, host):
        print("[+] CachedRespResolver. get_resps. host: {}".format(host))

        res = self._cache.get(host)
        if res:
            print("[+] CachedRespResolver. get_resps. resp from cache: {}".format(res))
            return res

        res = RespResolver.get_resps(host)
        self._cache[host] = res

        print("[+] CachedRespResolver. get_resps. resp new: {}".format(res))

        return res
