

from flask import Blueprint, render_template, request, redirect, url_for
from sqlalchemy import case

from app.utils import datetime_msk_2_timestamp_utc
from app.db.db import new_session
from app.db.models import DebbyScan, DebbyTask, DebbyPolicy
from app.db.models import DebbyAgent, DebbyProject
from app.settings import STATES_ALIVE, STATES_DEAD, STATE_IN_PROGRESS, STATE_CANCELED, STATE_PENDING
from app.tasks import log_scan_state
from app import yauth
from app.scans.utils import cancel_scan


bp = Blueprint('scans', __name__, url_prefix='/scans')


@bp.route('/')
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def scans_main():
    session = new_session()
    scans = session.query(DebbyScan, DebbyProject, DebbyPolicy).filter(
        DebbyScan.project_id == DebbyProject.id).filter(
        DebbyProject.policy_id == DebbyPolicy.id).order_by(
        DebbyScan.id.desc()).limit(100).all()
    session.close()

    for scan in scans:
        scan.DebbyProject.int_interval = datetime_msk_2_timestamp_utc(scan.DebbyProject.interval)

    return render_template('scans_main.html', scans=scans)


@bp.route('/<int:scan_id>', methods=['GET'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def scans_details(scan_id):

    try:
        offset_ = int(request.args.get('offset', 0))
        if offset_ < 0:
            raise ValueError
    except ValueError:
        offset_ = 0

    try:
        limit_ = int(request.args.get('limit', 20))
        if limit_ < 0:
            raise ValueError
    except ValueError:
        limit_ = 20

    session = new_session()

    cnt_tasks = session.query(DebbyTask).filter(DebbyTask.debbyscan_id == scan_id).count()

    cnt_sent_logs_tasks = session.query(DebbyTask).filter(DebbyTask.debbyscan_id == scan_id)\
                                                  .filter(DebbyTask.sent_logs == True).count()

    def order_func(task):
        return case(
            [
                (task.state == STATE_IN_PROGRESS, 0),
                (task.state == STATE_PENDING, 0),
            ],
            else_=1
        )

    tasks = session.query(DebbyTask).filter(DebbyTask.debbyscan_id == scan_id)\
                   .order_by(order_func(DebbyTask), DebbyTask.id.desc()).limit(limit_).offset(offset_).all()

    scan = session.query(DebbyScan, DebbyProject, DebbyPolicy).filter(DebbyScan.id == scan_id)\
                                                              .filter(DebbyScan.project_id == DebbyProject.id)\
                                                              .filter(DebbyProject.policy_id == DebbyPolicy.id).first()

    session.close()

    scan.DebbyProject.int_interval = datetime_msk_2_timestamp_utc(scan.DebbyProject.interval)

    scan_is_stopable = scan.DebbyScan.state not in STATES_DEAD

    return render_template('scans_details.html', scan=scan, tasks=tasks, scan_is_stopable=scan_is_stopable,
                           cnt_tasks=cnt_tasks, cnt_sent_logs_tasks=cnt_sent_logs_tasks)


# TODO: WTF here? ajax send tooooooooo many requests. Check it out
@bp.route('/<int:scan_id>', methods=['DELETE'])
@yauth.sessionid_required
# @yauth.admin_role_required
@yauth.require_role(yauth.ROLES_ADMIN)
def scan_stop(scan_id):
    is_canceled = cancel_scan(scan_id)
    if is_canceled:
        log_scan_state.s(scan_id, state=STATE_CANCELED).apply_async()
    return redirect(url_for('scans.scans_details', scan_id=scan_id))
