

import uuid

from app.agents.agentcli import new_agent_client
from app.db.db import safe_query, new_session, lock_safe_query
from app.db.models import DebbyTask, DebbyAgent, RetryTargets
from app.settings import STATE_PENDING, STATE_ABORTED, STATES_DEAD
from app.utils import eprint
from app.utils import now as utils_now


class TaskObject(object):
    def __init__(self, task_id):
        self._task_id = task_id

    def get_id(self):
        return self._task_id
    
    def get_targets(self):
        session = new_session()
        task = session.query(DebbyTask.targets).filter(DebbyTask.id == self._task_id).first()
        session.close()

        return task.targets if task else None

    def get_pdf_report(self):
        session = new_session()
        task = session.query(DebbyTask).filter(DebbyTask.id == self._task_id).first()
        agent = session.query(DebbyAgent).filter(DebbyAgent.id == task.debbyagent_id).first()
        session.close()

        # If agent not found/was deleted. Then cancel task.
        if not agent:
            def decrease_jobs_and_abort_task(s):
                s.query(DebbyAgent).filter(DebbyAgent.id == task.debbyagent_id)\
                                   .filter(DebbyAgent.jobs > 0).update({DebbyAgent.jobs: DebbyAgent.jobs - 1})
                s.query(DebbyTask).filter(DebbyTask.id == task.id).update({
                                                                    DebbyTask.state: STATE_ABORTED,
                                                                    DebbyTask.finish_time: utils_now()
                                                                })
            lock_safe_query(decrease_jobs_and_abort_task)
            return None

        # Get task info from agent
        cli = new_agent_client(agent)
        data = None
        try:
            data = cli.get_pdf_report(str(task.task_uuid))
        except NotImplementedError:
            return {"ok": False, "message": "NotImplementedError"}
        
        if not data:
            return {"ok": False, "message": "Unknown Error"}
        
        if data.get("status") != cli.STATUS_OK:
            return {"ok": False, "message": data.get('message')}
        
        if not data.get("content"):
            return {"ok": False, "message": "pdf report content is empty"}

        return {"ok": True, 'content': data.get("content")} 

        

    def fetch_task_info(self):
        """
        :return:
            {
              "engine": "nmap",
              "profile": "{u'args': u'-sS 5.255.255.5/32'}",        # engine specific profile
              "scan_stop_time": 1545669294,
              "scan_start_time": 1545669292,
              "state": "finished",
              "entry_create_time": 1545669292,                      # redundant
              "task_uuid": "c1f1dd5f-1d9e-4fb7-9f37-bc25a783b8c7",
              "results": [...],                                     # engine specific
              "save_to_db": false,
              "entry_modify_time": 1545669294                       # redundant
            }
        """

        # print("[TaskObject][fetch_task_info] Start. {}".format(self))

        session = new_session()
        task = session.query(DebbyTask).filter(DebbyTask.id == self._task_id).first()
        agent = session.query(DebbyAgent).filter(DebbyAgent.id == task.debbyagent_id).first()
        session.close()

        # If agent not found/was deleted. Then cancel task.
        if not agent:
            def decrease_jobs_and_abort_task(s):
                s.query(DebbyAgent).filter(DebbyAgent.id == task.debbyagent_id)\
                                   .filter(DebbyAgent.jobs > 0).update({DebbyAgent.jobs: DebbyAgent.jobs - 1})
                s.query(DebbyTask).filter(DebbyTask.id == task.id).update({
                                                                    DebbyTask.state: STATE_ABORTED,
                                                                    DebbyTask.finish_time: utils_now()
                                                                })
            lock_safe_query(decrease_jobs_and_abort_task)
            return None

        # Get task info from agent
        cli = new_agent_client(agent)
        # print("[TaskObject][fetch_task_info] task.task_uui:. {}".format(task.task_uuid))
        data = cli.get_task_info(str(task.task_uuid))
        # print("[TaskObject][fetch_task_info] data. {}".format(data))

        if not data:
            status = cli.STATUS_ERROR
        else:
            status = data['status']

        if status != cli.STATUS_OK:
            def decrease_jobs_and_abort_task(s):
                s.query(DebbyAgent).filter(DebbyAgent.id == task.debbyagent_id)\
                                   .filter(DebbyAgent.jobs > 0).update({DebbyAgent.jobs: DebbyAgent.jobs - 1})
                s.query(DebbyTask).filter(DebbyTask.id == task.id).update({
                                                                    DebbyTask.state: STATE_ABORTED,
                                                                    DebbyTask.finish_time: utils_now()
                                                                })
            lock_safe_query(decrease_jobs_and_abort_task)
            return None

        task_info = data['task_info']

        # Update task state if changed
        if task.state != task_info['state']:
            session = new_session()
            session.query(DebbyTask).filter(DebbyTask.id == self._task_id).update({DebbyTask.state: task_info['state']})
            session.commit()
            session.close()

        if task_info['state'] in STATES_DEAD:
            def decrease_jobs(sess):
                sess.query(DebbyTask).filter(DebbyTask.id == self._task_id)\
                                      .update({DebbyTask.finish_time: utils_now()})
                sess.query(DebbyAgent).filter(DebbyAgent.id == task.debbyagent_id)\
                                      .update({DebbyAgent.jobs: DebbyAgent.jobs - 1})
            lock_safe_query(decrease_jobs)

        return task_info


def try_run_task(scan_id, agent, payload, targets, retry_policy=None):
    """

    :param scan_id: 1297
    :param agent:   db.DebbyAgent object
    :param payload: {"engine": "nmap", "profile": {"args": "..."}, "save_to_db": false}
    :param targets: [u'5.255.255.5/32']
    :return:
        success: TaskObject object
        or
        failure: None
    """

    # run agent
    cli = new_agent_client(agent)
    r = cli.run_task(payload=payload)

    if not r:
        return None

    # Check that agent started successfully
    if r['status'] != cli.STATUS_OK:
        print('[+] try_run_task. agent: {}. payload: {}. response: {}'.format(agent.address, payload, r))

        # create task
        task_uuid = str(uuid.uuid4())
        dt = DebbyTask(debbyscan_id=scan_id, task_uuid=task_uuid, targets=' '.join(targets),
                       state=STATE_ABORTED, debbyagent_id=agent.id)

        def add_task(s):
            s.add(dt)
        safe_query(add_task)

        return None

    # create task
    task_uuid = r['task_uuid']
    dt = DebbyTask(debbyscan_id=scan_id, task_uuid=task_uuid, targets=' '.join(targets),
                   state=STATE_PENDING, debbyagent_id=agent.id)

    # increase jobs
    def add_task(s):
        s.add(dt)
        s.query(DebbyAgent).filter(DebbyAgent.id == agent.id).update({DebbyAgent.jobs: DebbyAgent.jobs + 1})
    lock_safe_query(add_task)

    # update retrytargets entry
    if retry_policy:
        def update_retry_targets(s):
            s.query(RetryTargets).filter(RetryTargets.id == retry_policy).update({RetryTargets.cur_task_id: dt.id})
        lock_safe_query(update_retry_targets)

    return TaskObject(dt.id)
