

from app.tasks.utils import get_events_from_db
from app.utils import now, datetime_msk_2_timestamp_utc
from app.db.db import new_session, lock_safe_query
from app.agents.agentcli import new_agent_client
from app.db.models import DebbyScan, DebbyTask, DebbyAgent, DebbyProject, PipelineProjects, ProjectApiInfo
from app.settings import STATES_ALIVE, STATE_CANCELED
from app.validators import DebbyOwnerException, DebbyRuntimeException
from app.settings import ENGINE_MOLLY
from app import infra


def cancel_scan(scan_id):
    """
    Cancel scan and related tasks by scan_id.
    :param scan_id:
    :return: None
    """
    session = new_session()
    scan = session.query(DebbyScan).filter(DebbyScan.id == scan_id).first()
    session.close()

    if not scan:
        return False

    if scan.state not in STATES_ALIVE:
        return False

    # Change scan state firstly, noy yo run other scans
    session = new_session()
    project = session.query(DebbyProject).filter(DebbyProject.id == scan.project_id).first()
    project_engine = project.engine
    session.query(DebbyScan).filter(DebbyScan.id == scan.id).update({DebbyScan.state: STATE_CANCELED,
                                                                     DebbyScan.finish_time: now()})
    session.commit()
    tasks = session.query(DebbyTask).filter(DebbyTask.debbyscan_id == scan_id) \
                                    .filter(DebbyTask.state.in_(STATES_ALIVE)).all()
    session.close()

    # remove infra info about crasher scan
    if project_engine == ENGINE_MOLLY:
        infra.stop_event(scan.id)

    for task in tasks:
        # get agent
        session = new_session()
        agent = session.query(DebbyAgent).filter(DebbyAgent.id == task.debbyagent_id).first()
        session.close()

        # send stop signal to agent
        # cli = AgentClient(agent)
        cli = new_agent_client(agent)
        cli.stop_task(task.task_uuid)

        def cancel_task_and_decrease_jobs(s):
            s.query(DebbyTask).filter(DebbyTask.id == task.id).update({DebbyTask.state: STATE_CANCELED})
            s.query(DebbyAgent).filter(DebbyAgent.id == task.debbyagent_id).filter(DebbyAgent.jobs > 0)\
                               .update({DebbyAgent.jobs: DebbyAgent.jobs - 1})

        lock_safe_query(cancel_task_and_decrease_jobs)

    return True


def get_pipeline_projects_by_prev_scan_id(scan_id):
    session = new_session()
    pipelines = session.query(PipelineProjects).filter(PipelineProjects.prev_proj_id == DebbyProject.id)\
                                               .filter(DebbyProject.id == DebbyScan.project_id)\
                                               .filter(DebbyScan.id == scan_id)\
                                               .all()
    session.close()
    return pipelines


def is_pipeline_projects_by_scan_id_for_prev_proj(scan_id):
    session = new_session()
    cnt = session.query(PipelineProjects).filter(PipelineProjects.prev_proj_id == DebbyProject.id)\
                                         .filter(DebbyProject.id == DebbyScan.project_id)\
                                         .filter(DebbyScan.id == scan_id)\
                                         .count()
    session.close()
    return cnt > 0


def check_api_src_permission_for_scan(api_src_id, scan_id):
    session = new_session()
    pai_cnt = session.query(ProjectApiInfo).filter(ProjectApiInfo.api_src_id == str(api_src_id))\
                                           .filter(ProjectApiInfo.project_id == DebbyProject.id)\
                                           .filter(DebbyProject.id == DebbyScan.project_id)\
                                           .filter(DebbyScan.id == scan_id).count()
    session.close()

    if pai_cnt == 0:
        raise DebbyOwnerException('Permission Denied')


def get_scan_details(scan_id, get_results=False):
    details = dict()

    session = new_session()

    scan = session.query(DebbyScan).filter(DebbyScan.id == scan_id).first()
    if not scan:
        session.close()
        raise DebbyRuntimeException("Not Found (scan_id: {})".format(scan_id))

    details["id"] = scan.id
    details["create_time"] = datetime_msk_2_timestamp_utc(scan.create_time) if scan.create_time else None
    details["finish_time"] = datetime_msk_2_timestamp_utc(scan.finish_time) if scan.finish_time else None
    details["state"] = scan.state

    if get_results:
        details["results"] = get_events_from_db(scan_id)

    session.close()

    return details
