

import os
import time
import datetime
import requests
from collections import Counter
from sqlalchemy import desc
from app.db.db import new_session
from app.db.models import DebbyAgent, DebbyProject, DebbyScan, DebbyTask, UnistatTable
from app.db.models import DebbyScanResults
from app.settings import STATES_ALIVE, STATES_DEAD
from app.settings import ENGINE_MOLLY
from app.utils import timestamp_utc_2_datetime_msk
from app.settings import STATE_CANCELED, STATE_FINISHED, STATE_ABORTED
from app.settings import SOLOMON_OAUTH_TOKEN
from app.settings import ENABLE_SOLOMON_MONITORING


def get_current_timestamp():
    return int(time.time())


def get_time_since_last_molly_run():
    s = new_session()

    molly_projects_ids = s.query(DebbyProject.id).filter(DebbyProject.deleted == False)\
                          .filter(DebbyProject.engine == ENGINE_MOLLY)\
                          .filter(DebbyProject.name == "molly2_stage2_molly")
    molly_last_scans = s.query(DebbyScan).filter(DebbyScan.project_id.in_(molly_projects_ids))\
                        .order_by(desc(DebbyScan.create_time)).first()

    s.close()

    if not molly_last_scans:
        return -1
    else:
        dt_since_last_molly_run = timestamp_utc_2_datetime_msk(get_current_timestamp()) - molly_last_scans.create_time
        seconds_since_last_molly_run = int(dt_since_last_molly_run.total_seconds())
        return seconds_since_last_molly_run


def get_molly_tasks_states_info():

    sensors = list()

    current_timestamp = get_current_timestamp()
    from_datetime = timestamp_utc_2_datetime_msk(current_timestamp) - datetime.timedelta(days=30)

    s = new_session()

    # get molly_tasks for last 30 days
    molly_projects_ids = s.query(DebbyProject.id).filter(DebbyProject.deleted == False)\
                          .filter(DebbyProject.engine == ENGINE_MOLLY)
    molly_scans_ids = s.query(DebbyScan.id).filter(DebbyScan.project_id.in_(molly_projects_ids))\
                       .filter(DebbyScan.create_time > from_datetime)
    molly_tasks = s.query(DebbyTask.state).filter(DebbyTask.debbyscan_id.in_(molly_scans_ids)).all()

    s.close()

    # counter every type of state
    c = Counter([task.state for task in molly_tasks])

    # prepare alive_state sensors
    for alive_state in STATES_ALIVE:
        counter =  c.get(alive_state, 0)
        sensors.append({
            "labels": {"sensor": alive_state.replace(" ", "_")},
            "ts": current_timestamp,
            "value": counter
        })


    s = new_session()
    # prepare dead_state sensors
    for dead_state in STATES_DEAD:

        counter =  c.get(dead_state, 0)

        prev_counter = s.query(UnistatTable)\
                        .filter(UnistatTable.key == dead_state)\
                        .first()

        # store prev value to make diff
        if not prev_counter:
            diff_counter = 0
            s.add(UnistatTable(key=dead_state, value=counter))
            s.commit()
        else:
            diff_counter = counter - prev_counter.value
            prev_counter.value = counter
            s.commit()

        sensors.append({
            "labels": {"sensor": dead_state.replace(" ", "_")},
            "ts": current_timestamp,
            "value": diff_counter
        })

    s.close()

    return sensors


def send_sensors(sensors, service, project="debby", cluster="console_prod"):
    if not ENABLE_SOLOMON_MONITORING:
        return

    url = "https://solomon.yandex.net/api/v2/push"
    params = {
        "project": project,
        "cluster": cluster,
        "service": service,
    }
    headers = {
        "Authorization": "OAuth {}".format(SOLOMON_OAUTH_TOKEN)
    }

    json_data = {
        "commonLabels": {
            "host": os.getenv("QLOUD_DISCOVERY_COMPONENT", "solomon-push")
        },
        "sensors": sensors
    }

    resp = requests.post(url, params=params, headers=headers, json=json_data)
    resp_json = resp.json()
    if resp_json.get("status") != "OK":
        print("[send_sensors] Status not OK: {}".format(resp.text))


def get_sensors_for_last_molly_run():
    seconds_elapsed = get_time_since_last_molly_run()

    return [{
        "labels": {"sensor": "seconds_since_last_molly_run"},
        "ts": get_current_timestamp(),
        "value": seconds_elapsed
    }]


def get_sensors_for_last_molly_task_run():
    s = new_session()
    molly_projects_ids = s.query(DebbyProject.id).filter(DebbyProject.deleted == False)\
                      .filter(DebbyProject.engine == ENGINE_MOLLY)\
                      .filter(DebbyProject.name == "molly2_stage2_molly")
    molly_last_scans_ids = s.query(DebbyScan.id).filter(DebbyScan.project_id.in_(molly_projects_ids))\
                        .order_by(desc(DebbyScan.create_time))
    molly_last_task = s.query(DebbyTask).filter(DebbyTask.debbyscan_id.in_(molly_last_scans_ids))\
                        .order_by(desc(DebbyTask.create_time)).first()
    s.close()

    seconds_elapsed = 100500
    if molly_last_task:
        cur_msk_time = timestamp_utc_2_datetime_msk(time.time())
        seconds_elapsed = (cur_msk_time - molly_last_task.create_time).total_seconds()

    return [{
        "labels": {"sensor": "seconds_since_last_molly_task_run"},
        "ts": get_current_timestamp(),
        "value": seconds_elapsed
    }]



def solomon_routine():
    if not ENABLE_SOLOMON_MONITORING:
        return

    # time measurement
    start_time = time.time()

    try:
        molly_tasks_sensors = get_molly_tasks_states_info()
        send_sensors(molly_tasks_sensors, "molly_tasks_states")
    except AttributeError as e:
        print("EXCEPTION. solomon_routine. molly_tasks_sensors. {}".format(e))

    try:
        molly_last_scan_seconds_sensors = get_sensors_for_last_molly_run()
        send_sensors(molly_last_scan_seconds_sensors, "molly_last_scan_run")
    except AttributeError as e:
        print("EXCEPTION. solomon_routine. molly_last_scan_seconds_sensors. {}".format(e))

    try:
        molly_last_task_seconds_sensors = get_sensors_for_last_molly_task_run()
        send_sensors(molly_last_task_seconds_sensors, "molly_last_task_run")
    except AttributeError as e:
        print("EXCEPTION. solomon_routine. molly_last_task_seconds_sensors. {}".format(e))

    # time measurement
    finish_time = time.time()
    timeout = int(round((finish_time - start_time) * 1000))
    timeout_sensors = [{
        "labels": {"sensor": "timeout_ms"},
        "ts": get_current_timestamp(),
        "value": timeout
    }]
    send_sensors(timeout_sensors, "solomon_routine_timeout_ms")


def count_molly_tasks_runs(dt_from, dt_till):
    s = new_session()
    molly_projects_ids = s.query(DebbyProject.id).filter(DebbyProject.deleted == False)\
                          .filter(DebbyProject.engine == ENGINE_MOLLY)\
                          .filter(DebbyProject.name == "molly2_stage2_molly")
    molly_last_scans_ids = s.query(DebbyScan.id).filter(DebbyScan.project_id.in_(molly_projects_ids))\
                            .order_by(desc(DebbyScan.create_time))
    tasks = s.query(DebbyTask.state)\
             .filter(DebbyTask.debbyscan_id.in_(molly_last_scans_ids))\
             .filter(DebbyTask.state.in_(STATES_DEAD))\
             .filter(DebbyTask.create_time > dt_from)\
             .filter(DebbyTask.create_time <= dt_till).all()
    s.close()

    c = Counter([task.state for task in tasks])

    tasks_finished = c.get(STATE_FINISHED, 0)
    tasks_aborted = c.get(STATE_ABORTED, 0)
    tasks_canceled = c.get(STATE_CANCELED, 0)
    tasks_total = tasks_finished + tasks_aborted + tasks_canceled

    return {
        "total": tasks_total,
        "finished": tasks_finished,
        "aborted": tasks_aborted,
        "canceled": tasks_canceled,
    }


def solomon_daily_routine():
    # time measurement
    start_time = time.time()

    # molly tasks
    ts = get_current_timestamp()
    msk_dt = timestamp_utc_2_datetime_msk(ts)
    ts_till = ts - msk_dt.minute*60 - msk_dt.second
    dt_till = timestamp_utc_2_datetime_msk(ts_till)
    dt_from = dt_till - datetime.timedelta(days=1)

    molly_tasks_runs = count_molly_tasks_runs(dt_from, dt_till)
    molly_daily_sensors = list([{
            "labels": {"sensor": "daily_molly_{}".format(state)},
            "ts": ts,
            "value": molly_tasks_runs.get(state, 0)
        } for state in ["total", "finished", "aborted", "canceled"]])
    send_sensors(molly_daily_sensors, "molly_daily_results")

    # time measurement
    finish_time = time.time()
    timeout = int(round((finish_time - start_time) * 1000))
    timeout_sensors = [{
        "labels": {"sensor": "daily_timeout_ms"},
        "ts": get_current_timestamp(),
        "value": timeout
    }]
    send_sensors(timeout_sensors, "solomon_daily_timeout_ms")


def get_nmap_found_ips_for_molly(dt_from, dt_till):
    s = new_session()
    nmap_projects_ids = s.query(DebbyProject.id).filter(DebbyProject.deleted == False)\
                         .filter(DebbyProject.name == "molly2_stage1_nmap")
    nmap_last_scans_ids = s.query(DebbyScan.id)\
                           .filter(DebbyScan.project_id.in_(nmap_projects_ids))\
                           .filter(DebbyTask.create_time > dt_from)\
                           .filter(DebbyTask.create_time <= dt_till)\
                           .order_by(desc(DebbyScan.create_time)).limit(5)
    nmap_scan_results = s.query(DebbyScanResults)\
                         .filter(DebbyScanResults.scan_id.in_(nmap_last_scans_ids)).count()

    s.close()

    return nmap_scan_results


def solomon_weekly_routine():
    # time measurement
    start_time = time.time()

    # molly tasks
    ts = get_current_timestamp()
    msk_dt = timestamp_utc_2_datetime_msk(ts)
    ts_till = ts - msk_dt.minute*60 - msk_dt.second
    dt_till = timestamp_utc_2_datetime_msk(ts_till)
    dt_from = dt_till - datetime.timedelta(days=7)

    molly_tasks_runs = count_molly_tasks_runs(dt_from, dt_till)
    molly_weekly_sensors = list([{
            "labels": {"sensor": "weekly_molly_{}".format(state)},
            "ts": ts,
            "value": molly_tasks_runs.get(state, 0)
        } for state in ["total", "finished", "aborted", "canceled"]])
    send_sensors(molly_weekly_sensors, "molly_weekly_results")

    nmap_found_ips_for_molly = get_nmap_found_ips_for_molly(dt_from, dt_till)
    nmap_for_molly_weekly_sensors = [{
        "labels": {"sensor": "weekly_nmap_found_ips_for_molly"},
        "ts": get_current_timestamp(),
        "value": nmap_found_ips_for_molly
    }]
    send_sensors(nmap_for_molly_weekly_sensors, "weekly_nmap_for_molly")

    # time measurement
    finish_time = time.time()
    timeout = int(round((finish_time - start_time) * 1000))
    timeout_sensors = [{
        "labels": {"sensor": "weekly_timeout_ms"},
        "ts": get_current_timestamp(),
        "value": timeout
    }]
    send_sensors(timeout_sensors, "solomon_weekly_timeout_ms")
