

import ipaddress
import re
import socket
import traceback
from dns.resolver import NXDOMAIN, NoAnswer, Resolver

from app.macros.macro import MacroBaseObject
from app.utils import is_valid_ipv4_address, is_valid_ipv6_address, is_subnet


class InputTargetType:
    """
    Convertions:
        FQDN -> IPv4, IPv6
        IPv6 -> FQDN
        IPv4 -> FQDN
        Macro -> FQDN, IPv4, IPv6, Subnet, (Macro?)
        Subnet -> IPv4, IPv6
        FromProject -> IPv4, IPv6
    """

    FQDN = "FQDN"       # "ya.ru"
    IPv6 = "IPV6"       # "202a::23"
    IPv4 = "IPV4"       # "192.168.1.1"
    Macro = "MACRO"     # "_YANDEXNETS_"
    Subnet = "SUBNET"   # "192.168.1.0/24", "202a::ff00/125"
    FromProject = "FROM_PROJECT"    # "#FROM_PROJECT#123#1-79,81-442,444-65535#"
    # Unknown = "UNKNOWN"


class ResultingTargetType:
    IP_OR_FQDN = "IP_OR_FQDN"
    IP = "IP"
    VHOSTS = "VHOSTS"


class TargetResolver(object):

    @staticmethod
    def guess_type(value):
        if is_valid_ipv4_address(value):
            return InputTargetType.IPv4
        elif is_valid_ipv6_address(value):
            return InputTargetType.IPv6
        elif is_subnet(value):
            return InputTargetType.Subnet
        # elif re.match(r"#FROM_PROJECT#\d+#[\d,-]*#", value):
        #     return InputTargetType.FromProject
        elif value and value[0] == "_" and value[-1] == "_" and "." not in value:
            return InputTargetType.Macro
        else:
            return InputTargetType.FQDN


    @staticmethod
    def resolve_macro(target_macro):
        target_macro = target_macro.strip()
        mbo = MacroBaseObject(target_macro)
        targets = mbo.get_targets()
        for target in targets:
            yield target

    @staticmethod
    def resolve_subnet(target_subnet):
        target_subnet = target_subnet.strip()

        try:
            net = ipaddress.ip_network(target_subnet)
            for host in net.hosts():
                yield str(host)

        except Exception:
            print("[!] TargetResolver. resolve_subnet. EXCEPTION:")
            traceback.print_exc()

    @staticmethod
    def resolve_fqdn(target_fqdn):
        target_fqdn = target_fqdn.strip()

        try:
            dns_resolver = Resolver()
        except NXDOMAIN:
            return

        try:
            ipv4_answer = dns_resolver.query(target_fqdn, "A")
        except (NoAnswer, NXDOMAIN):
            ipv4_answer = []
        for ipv4 in list([str(x) for x in ipv4_answer]):
            yield ipv4

        try:
            ipv6_answer = dns_resolver.query(target_fqdn, "AAAA")
        except (NoAnswer, NXDOMAIN):
            ipv6_answer = []
        for ipv6 in list([str(x) for x in ipv6_answer]):
            yield ipv6

    @staticmethod
    def ip_to_fqdn(target_ip):
        target_ip = target_ip.strip()

        try:
            return socket.gethostbyaddr(target_ip)[0]
        except socket.herror:
            return None

    @staticmethod
    def resolve_one_to_ip_or_fqdn(target):
        target = target.strip()

        # ipv4 or ipv6
        if TargetResolver.guess_type(target) in [InputTargetType.IPv6, InputTargetType.IPv4]:
            yield target
            return

        # subnet
        if is_subnet(target):
            for resolved in TargetResolver.resolve_subnet(target):
                yield resolved
            return

        # fqdn -> fqdn
        if '.' in target:
            yield target
            return

        # macros
        if len(target) and target[0] == '_' and target[-1] == '_':
            for resolved in TargetResolver.resolve_macro(target):
                for recursive_resolved in TargetResolver.resolve_one_to_ip_or_fqdn(resolved):
                    yield recursive_resolved
            return

        # filters random staff like "localhost", "domainwodots"
        return

    # @staticmethod
    # def resolve_many_to_ip_or_fqdn(targets):
    #     for target in targets:
    #         for resolved in TargetResolver.resolve_one_to_ip_or_fqdn(target):
    #             yield resolved

    @staticmethod
    def resolve_one(target, resolving_target_type):
        if resolving_target_type == ResultingTargetType.IP_OR_FQDN:
            for resolved in TargetResolver.resolve_one_to_ip_or_fqdn(target):
                yield resolved
        else:
            return

    @staticmethod
    def resolve_many_to_ip_or_fqdn(targets, resolving_target_type):
        for target in targets:
            for resolved in TargetResolver.resolve_one(target, resolving_target_type):
                yield resolved

