

import uuid
import time
import datetime
import traceback
import logging
import smtplib
import tempfile
from sqlalchemy import or_
from sqlalchemy import delete
from celery import Celery
from psycopg2.extensions import ISOLATION_LEVEL_AUTOCOMMIT

from app.db.db import new_session, safe_query, get_engine
from app.db.models import DebbyTask, DebbyScan, DebbyTag, PipelineScans, DebbyScanResults, DebbyScanResultsScripts
from app.db.models import DebbyScanResultsService, ProjectApiInfo, DebbyAgent
from app.db.models import RelationProjectTag
from app.db.models import DebbyPolicy, DebbyProject
from app.db.models import PortCache, RetryTargets
from app.engines import new_engine
from app import infra
from app.macros.utils import fetch_golem_macros
# from app.projects.utils import get_scheduled_projects
from app.settings import STATE_IN_PROGRESS, STATE_FINISHED, STATE_CANCELED, STATE_ABORTED
from app.settings import EVENT_TYPE_STATE_CHANGE
from app.settings import STATES_ALIVE, STATES_DEAD, STATES_FAILED, STATE_TYPE_ALIVE, STATE_TYPE_DEAD, STATE_TYPE_UNKNOWN
from app.settings import ENGINE_MOLLY, ENGINE_OPENVAS_2
from app.scans.task import try_run_task, TaskObject
from app.scans.utils import cancel_scan, is_pipeline_projects_by_scan_id_for_prev_proj
from app.scans.utils import get_pipeline_projects_by_prev_scan_id
from app.tasks.splunk import send_log_to_splunk, send_event
from app.tasks.utils import get_available_agent_2, save_events_to_db
from app.utils import datetime_msk_2_timestamp_utc, now
from app.utils import eprint
from app.validators import DebbyException
from app.solomon import solomon_routine, solomon_daily_routine, solomon_weekly_routine
from app.settings import FEATURE_ENABLE_SEND_OPENVAS_REPORT_TO_ST
from startrek_client import Startrek
from startrek_client.exceptions import NotFound, Forbidden
from app.settings import ROBOT_ST_OAUTH_TOKEN, ST_DEBBY_CONSOLE_USER_AGENT

celery_app = Celery()
celery_app.config_from_object('app.tasks.celeryconfig')


@celery_app.task
def update_working_tasks_info():

    # Get Tasks list
    session = new_session()
    tasks = session.query(DebbyTask).filter(DebbyTask.state.in_(STATES_ALIVE))\
                   .filter(or_(
                        DebbyTask.sent_to_analysis == None,
                        DebbyTask.sent_to_analysis < now() - datetime.timedelta(minutes=10)
                   )).all()

    # update sent_to_analysis time
    # and get ids
    now_ = now()
    task_id_list = list()
    for task in tasks:
        task.sent_to_analysis = now_
        task_id_list.append(task.id)

    session.commit()
    session.close()

    if task_id_list:
        update_working_tasks_list.delay(task_id_list)


@celery_app.task
def update_working_tasks_list(task_id_list):

    # print("[update_working_tasks_list] task_id_list:{}.".format(task_id_list))
    # TODO: Parallelize
    for task_id in task_id_list:
        # print("[update_working_tasks_list] task_id:{}.".format(task_id))

        task_obj = TaskObject(task_id)
        # print("[update_working_tasks_list] task_obj:{}.".format(task_obj))
        task_info = task_obj.fetch_task_info()

        # print("[update_working_tasks_list] task_info:{}.".format(task_info))

        if not task_info:
            task_state = STATE_ABORTED
            task_results = None
        else:
            task_state = task_info['state']
            task_results = task_info['results']

        # if task_state not in STATES_DEAD:
        #     session = new_session()
        #     session.query(DebbyTask).filter(DebbyTask.id == task_id).
        #     session.close()

        if task_state in STATES_DEAD:

            # get scan and project objects
            session = new_session()
            task = session.query(DebbyTask).filter(DebbyTask.id == task_id).first()
            scan = session.query(DebbyScan).filter(DebbyScan.id == task.debbyscan_id).first()
            project = session.query(DebbyProject).filter(DebbyProject.id == scan.project_id).first()
            pai = session.query(ProjectApiInfo).filter(ProjectApiInfo.project_id == project.id).first()
            session.close()

            # clean up task for engine
            engine = new_engine(project.engine)
            engine.task_cleanup(task)

            splunk_events = list()
            # Send logs to splunk
            if task_results:
                print("task_results: scan.id={}, task_results={}".format(scan.id, task_results))
                engine = new_engine(project.engine)

                # Cache ports for nmaps similar tasks
                try:
                    cached_count = engine.cache_ports(task_results, project.name, scan.id, task_id, task.debbyagent_id)
                    # if cached_count:
                    #     print("[+] Cached {} results for project={} scan={}, task={}".format(cached_count, project.name, scan.id, task_id))
                except Exception as e:
                    print("[!] Exception occured during caching results for project={} scan={}, task={}. e={}".format(project.name, scan.id, task_id, e))
                
                # Prepare and send results to splunk
                all_splunk_events = engine.scan_results_to_splunk_events(task_results, task_id)
                if project.log_closed == True:
                    splunk_events = all_splunk_events
                else:
                    splunk_events = list([event for event in all_splunk_events if event.get('enabled') is True])
                send_splunk_logs.delay(splunk_events, task_id, scan.id)
                if FEATURE_ENABLE_SEND_OPENVAS_REPORT_TO_ST and project.engine == ENGINE_OPENVAS_2 and project.st_ticket:
                    send_openvas_report_to_st.delay(task_id, project.st_ticket)
            else:
                print("AGENT RETURNED EMPTY RESULTS. task_id: {}. scan_id: {}. project_name: {}".format(task_id, scan.id, project.name))

            # Save to db if pipelined or API call
            if is_pipeline_projects_by_scan_id_for_prev_proj(scan.id) or (pai is not None):
                save_events_to_db(splunk_events)

            session = new_session()
            session.query(PipelineScans).filter(PipelineScans.next_scan_id == scan.id).update({
                PipelineScans.state: STATE_TYPE_DEAD
            })
            session.commit()
            session.close()

            # notify responsible about failed tasks
            def notifyOnFail():
                print("[notifyOnFail]")
                contacts = project.contacts.split(",") if project.contacts else ''
                contacts = list(map(lambda x:x.strip(), contacts))
                contacts = list(map(lambda x:x.split("@")[0] + "@yandex-team.ru", contacts))
                contacts = list(set(contacts))
                if not contacts:
                    return

                EMAIL_HOST = "outbound-relay.yandex.net"
                EMAIL_PORT = 25
                EMAIL_SENDER = 'robot-debby@yandex-team.ru'
                EMAIL_RECEIVERS = contacts

                s=smtplib.SMTP(EMAIL_HOST, EMAIL_PORT)
                msg  = '\\\n'
                msg += "From: robot-debby@yandex-team.ru\n"
                msg += "Subject: ~~~DEBBY ALERT~~~ Retries Failed ~~~DEBBY ALERT~~~ \n\n"
                msg += "Failed retries for target: '{}'\n".format(task.targets)
                msg += "Look at scan: https://debby-console.sec.yandex-team.ru/scans/{}.  task_id: {}\n".format(scan.id, task.id)
                s.sendmail(EMAIL_SENDER,EMAIL_RECEIVERS,msg)
                s.quit()

            # retry on fails
            def retryOnFail():
                # if project.retries == 0:
                #     return
                # print("[retryOnFail] start")
                if not task_state in STATES_FAILED:
                    return
                session = new_session()
                r = session.query(RetryTargets).filter(RetryTargets.cur_task_id == task.id).first()
                session.close()
                failed_count = r.failed_count if r else 0
                if not r:
                    if project.retries == 0:
                        # print("[retryOnFail] No existing RetryTargets and project.retries={}, exiting.".format(project.retries))
                        return
                    rt = RetryTargets(scan_id=scan.id, failed_task_id=None, cur_task_id=task.id, failed_count=0)
                    # print("[retryOnFail] No existing RetryTargets for task_id {}. Creater new one: {}".format(task.id, rt))
                    session = new_session()
                    session.add(rt)
                    session.commit()
                    session.close()
                    # return
                if failed_count >= project.retries:
                    if project.retries != 0:
                        notifyOnFail()
                    session = new_session()
                    stmt = delete(RetryTargets).where(RetryTargets.cur_task_id == task.id)
                    # print("[retryOnFail] removing. stmt:{}".format(stmt))
                    session.execute(stmt)
                    session.commit()
                    session.close()
                else:
                    session = new_session()
                    session.query(RetryTargets).filter(RetryTargets.cur_task_id == task.id)\
                            .update({
                                RetryTargets.failed_count: RetryTargets.failed_count+1,
                                RetryTargets.failed_task_id: task.id
                            })
                    session.commit()
                    session.close()
                    # print("[retryOnFail] increased failed_count")
              
            retryOnFail()

            # Update scan status
            if scan.all_targets_tasked and scan.state not in STATES_DEAD:

                # get tasks of current scan
                session = new_session()
                tasks = session.query(DebbyTask.state).filter(DebbyTask.debbyscan_id == scan.id).all()
                session.close()

                # get booleans of active states
                not_active_states_bools = list([task.state in STATES_DEAD for task in tasks])

                # change state if all tasks' states are not active
                if all(not_active_states_bools):
                    if project.engine == ENGINE_MOLLY:
                        infra.stop_event(scan.id)
                    session = new_session()
                    session.query(DebbyScan).filter(DebbyScan.id == scan.id)\
                           .update({
                                DebbyScan.state: STATE_FINISHED,
                                DebbyScan.finish_time: now()
                            })
                    session.commit()
                    session.close()

                    # scan_cleanup
                    session = new_session()
                    required_tags = session.query(DebbyTag.value).filter(RelationProjectTag.tag_id == DebbyTag.id)\
                                                                 .filter(RelationProjectTag.project_id == project.id).all()
                    session.close()
                    required_tags_list = list([x.value for x in required_tags])
                    engine.scan_cleanup(scan.id, required_tags_list)

                    # run all pipelines if exists
                    # print('[+] update_working_tasks_list. scan.id: {}'.format(scan.id))
                    for pipeline in get_pipeline_projects_by_prev_scan_id(scan.id):
                        # print('[+] update_working_tasks_list. pipeline.next_proj_id: {}'.format(pipeline.next_proj_id))
                        run_scan_for_project.delay(pipeline.next_proj_id, prev_scan_id=scan.id)

                    # remove prev scan data if no more active scans
                    session = new_session()
                    cur_pipeline = session.query(PipelineScans).filter(PipelineScans.next_scan_id == scan.id).first()
                    if cur_pipeline:
                        pipeline_prev_scan_id = cur_pipeline.prev_scan_id
                        all_prev_scan_pipelines = session.query(PipelineScans)\
                                                         .filter(PipelineScans.prev_scan_id == pipeline_prev_scan_id)\
                                                         .all()

                        cnt = 0
                        for pipeline in all_prev_scan_pipelines:
                            if pipeline.state == STATE_TYPE_ALIVE:
                                cnt += 1

                        if cnt == 0:
                            session.query(DebbyScanResultsScripts)\
                                   .filter(DebbyScanResultsScripts.scan_result_id == DebbyScanResults.id)\
                                   .filter(DebbyScanResults.scan_id == pipeline_prev_scan_id)\
                                   .delete(synchronize_session='fetch')
                            session.query(DebbyScanResultsService)\
                                   .filter(DebbyScanResultsService.scan_result_id == DebbyScanResults.id)\
                                   .filter(DebbyScanResults.scan_id == pipeline_prev_scan_id)\
                                   .delete(synchronize_session='fetch')
                            session.query(DebbyScanResults)\
                                   .filter(DebbyScanResults.scan_id == pipeline_prev_scan_id)\
                                   .delete(synchronize_session='fetch')
                            session.query(PipelineScans)\
                                   .filter(PipelineScans.prev_scan_id == pipeline_prev_scan_id)\
                                   .delete(synchronize_session='fetch')
                            session.commit()

                    session.close()

        def sent_to_analysis_update(sess):
            task = sess.query(DebbyTask).filter(DebbyTask.id == task_id).first()
            task.sent_to_analysis = None
        safe_query(sent_to_analysis_update)

    # def sent_to_analysis_update(sess):
    #     for task_id in task_id_list:
    #         task = sess.query(DebbyTask).filter(DebbyTask.id == task_id).first()
    #         task.sent_to_analysis = None
    # safe_query(sent_to_analysis_update)


@celery_app.task
def send_openvas_report_to_st(task_id, st_ticket):
    task_obj = TaskObject(task_id)
    task_info = task_obj.get_pdf_report()
    res = task_obj.get_pdf_report()
    if not res.get("ok"):
        print("[!] Unable to get pdf report. task={}. msg={}".format(task_id, res.get('message')))
        return
    
    targets = task_obj.get_targets()

    suffix = "_"
    if all(x not in targets for x in [" ", ","]):
        suffix += targets + "_"  # yandex.ru
    suffix += datetime.datetime.now().strftime("%Y-%m-%d:%H:%M:%S")  # '2022-06-03:01:34:21'
    suffix += "_report.pdf"
    # ex: _yandex.ru_2022-06-03:01:34:21_report.pdf

    f = tempfile.NamedTemporaryFile(suffix=suffix, delete=False)
    f.write(res.get('content'))
    f.close()

    try:
        client = Startrek(token=ROBOT_ST_OAUTH_TOKEN, useragent=ST_DEBBY_CONSOLE_USER_AGENT)
        issue = client.issues[st_ticket]
        comment = issue.comments.create(text='OpenVAS report for %%{}%%'.format(targets), attachments=[f.name])
    except NotFound as e:
        print("Ticket {} for task {} not found. err={}".format(st_ticket, task_id, e))
    except Forbidden as e:
        print("Access to ticket {} is forbidden for task {}. err={}".format(st_ticket, task_id, e))
    except Expection as e:
        print("Some exception occured during a processing ticket {} for task {}. err={}".format(st_ticket, task_id, e))

    os.remove(f.name)

    return
    # session = new_session()
    # tasks = session.query(DebbyTask).filter(DebbyTask.id == task_id).first()
    # project = session.query(DebbyScan).filter(DebbyTask.debbyscan_id == scan_id).all()
    # session.close()

@celery_app.task
def send_splunk_logs(splunk_events, task_id, scan_id, retries=3):

    # print('[+] send_splunk_logs. splunk_events: {}.'.format(splunk_events))

    # send logs if present with retries
    if splunk_events:

        ok = send_log_to_splunk(splunk_events, task_id, scan_id)

        if not ok:
            print('[+] tasks. send_splunk_logs. SPLUNK not sent. scan_id:{}. task_id:{}.'.format( scan_id, task_id))
            return

    # update log status
    session = new_session()
    session.query(DebbyTask).filter(DebbyTask.id == task_id).update({DebbyTask.sent_logs: True})
    session.commit()
    session.close()

    # get all sent logs info for current scan
    session = new_session()
    tasks = session.query(DebbyTask.sent_logs).filter(DebbyTask.debbyscan_id == scan_id).all()
    session.close()

    # check if all logs sent
    all_finished = True
    for task in tasks:
        if not task.sent_logs:
            all_finished = False
            break

    if all_finished:
        session = new_session()
        session.query(DebbyScan).filter(DebbyScan.id == scan_id).update({DebbyScan.log_finish_time: now()})
        session.commit()
        session.close()

        log_scan_state.s(scan_id, state=STATE_FINISHED).apply_async(countdown=1)


def get_scheduled_projects():
    """
    Get list of project ids that are ready to run scan
    Algorithg:
        1. Get all active projects
        2. Get all project which (OR)
        2.1. didnt run before
        2.2. preveously were runned more thatn period ago
    :return:
    """

    cur_time = now()

    session = new_session()
    projects = session.query(DebbyProject).filter(DebbyProject.deleted == False)\
                                          .filter(DebbyProject.paused == False)\
                                          .filter(cur_time < DebbyProject.scan_stop)\
                                          .filter(cur_time >= DebbyProject.scan_start).all()
    session.close()

    for project in projects:
        ts_interval = datetime_msk_2_timestamp_utc(project.interval)

        if ts_interval == 0:
            if not project.last_scan:
                yield project.id

        else:
            diff = cur_time - project.scan_start
            seconds_diff = int(diff.total_seconds())
            over_time = seconds_diff % (ts_interval*60)
            max_over_time = 30

            if not project.last_scan:
                last_scan_diff = None
            else:
                last_scan_diff = int((cur_time - project.last_scan).total_seconds())

            if over_time < max_over_time:
                if last_scan_diff is None or last_scan_diff > max_over_time:
                    yield project.id

@celery_app.task
def run_scheduled_projects():
    """
    Algorithm:
    1. Get all scheduled projects
    2. Run scan for project
    """

    for project_id in get_scheduled_projects():
        run_scan_for_project.delay(project_id)


@celery_app.task
def run_scan_for_project(project_id, prev_scan_id=None):
    """
    Algorithm:
    1. Get required tags
    2. Check if agent exists
    3. Create scan object on succes
    4. Get policy object for that project's scan
    5. For every target:
    5.1. Transform policy to args
    5.2. Check that scan not canceled
    5.3. Run scan
    5.4. Create task entry
    5.5. Inscrease agent's jobs counter
    6. Update scan info
    """

    ds_id = None
    project_engine = None

    # get required tags list
    session = new_session()
    required_tags = session.query(DebbyTag.value).filter(RelationProjectTag.tag_id == DebbyTag.id)\
                                                 .filter(RelationProjectTag.project_id == project_id).all()
    session.close()

    required_tags_list = list([x.value for x in required_tags])

    try:

        # check for agents that meet the conditions
        if not get_available_agent_2(required_tags_list, any_job_count=True):
            if prev_scan_id:
                # print('[+] run_scan_for_project. STATE_TYPE_DEAD')
                ps = PipelineScans(prev_scan_id=prev_scan_id, state=STATE_TYPE_DEAD)
                session = new_session()
                session.add(ps)
                session.commit()
                session.close()
            return

        # get active scan and stop it
        session = new_session()
        alive_scans = session.query(DebbyScan).filter(DebbyScan.project_id == project_id)\
                                              .filter(DebbyScan.state.in_(STATES_ALIVE)).all()
        project = session.query(DebbyProject).filter(DebbyProject.id == project_id).first()
        project_engine = project.engine
        session.close()
        for alive_scan in alive_scans:
            if project_engine == ENGINE_MOLLY:
                infra.stop_event(alive_scan.id)
            if cancel_scan(alive_scan.id):
                log_scan_state.s(alive_scan.id, state=STATE_CANCELED).apply_async(countdown=1)

        cur_time = now()
        # create new scan. update project last scan time
        session = new_session()

        project = session.query(DebbyProject).filter(DebbyProject.id == project_id).first()
        project_engine = project.engine

        # time
        ts_interval = datetime_msk_2_timestamp_utc(project.interval)
        if ts_interval == 0:
            time_to_save = cur_time
        else:
            diff = cur_time - project.scan_start
            seconds_diff = int(diff.total_seconds())
            over_time = seconds_diff % (ts_interval*60)
            time_to_save = cur_time - datetime.timedelta(seconds=over_time)

        project.last_scan = time_to_save

        ds = DebbyScan(project_id=project_id, state=STATE_IN_PROGRESS)
        session.add(ds)
        session.commit()
        ds_id = ds.id

        if prev_scan_id:
            ps = PipelineScans(prev_scan_id=prev_scan_id, next_scan_id=ds_id, state=STATE_TYPE_ALIVE)
            session.add(ps)
            session.commit()

        # log splunk scan state
        if project_engine == ENGINE_MOLLY:
            infra.start_event(ds_id)
        log_scan_state.s(ds_id, state=STATE_IN_PROGRESS).apply_async(countdown=1)

        session.close()

        # ------------------
        # ------------------
        # ------------------

        engine = new_engine(project_engine)
        engine.scan_init(ds_id, required_tags_list, project_id)

        for task_payload in engine.new_tasks_payloads_generator(project_id, scan_id=ds_id):

            targets = None
            payload = None
            retry_policy = None
            try:
                targets, payload, retry_policy = task_payload
            except ValueError:
                targets, payload = task_payload

            is_scan_dead = False

            retries = 5
            while True:
                # Break if canceled
                scan = safe_query(lambda s: s.query(DebbyScan).filter(DebbyScan.id == ds_id).first())
                if scan.state in STATES_DEAD:
                    is_scan_dead = True
                    break

                # get available policies
                agent = get_available_agent_2(required_tags_list)
                if not agent:
                    time.sleep(0.1)
                    continue

                # Catch exception ob initialization.
                # Openvas may raise cause of unstable copy-paste policy proxy
                try:
                    task_meta_info = engine.task_init(ds_id, agent, payload, targets)
                except DebbyException as e:
                    now_dt = now()
                    dt = DebbyTask(debbyscan_id=ds_id, task_uuid=str(uuid.uuid4()), 
                                   targets=' '.join(targets), state=STATE_ABORTED, 
                                   debbyagent_id=agent.id, finish_time=now_dt,
                                   create_time=now_dt)
                    safe_query(lambda s: s.add(dt))
                    print('[!] Task aborted on task_init operation')
                    traceback.print_exc()
                    break

                payload = engine.task_pre_run(payload, task_meta_info)
                task_object = try_run_task(ds_id, agent, payload, targets, retry_policy=retry_policy)

                if task_object:
                    debby_task_id = task_object.get_id()
                    engine.task_post_run(debby_task_id, task_meta_info)
                    break
                
                # up to 5 retries for aborted responses
                if retries == 0:
                    break
                else:
                    retries -= 1

            if is_scan_dead:
                break

        def proj_run_ending(s):
            # if 0 valid targets then finished
            cnt = s.query(DebbyTask).filter(DebbyTask.debbyscan_id == ds_id).count()

            if cnt == 0:
                s.query(DebbyScan).filter(DebbyScan.id == ds_id).update({DebbyScan.state: STATE_FINISHED})
                if project_engine == ENGINE_MOLLY:
                    infra.stop_event(ds_id)
                log_scan_state.s(ds_id, state=STATE_FINISHED).apply_async(countdown=1)
                engine.scan_cleanup(ds_id, required_tags_list)

            # check if all tasks are aborted 
            else:
                aborted_cnt = s.query(DebbyTask).filter(DebbyTask.debbyscan_id == ds_id)\
                                        .filter(DebbyTask.state == STATE_ABORTED).count()

                if cnt == aborted_cnt:
                    s.query(DebbyScan).filter(DebbyScan.id == ds_id).update({DebbyScan.state: STATE_ABORTED})
                    if project_engine == ENGINE_MOLLY:
                        infra.stop_event(ds_id)
                    log_scan_state.s(ds_id, state=STATE_ABORTED).apply_async(countdown=1)
                    engine.scan_cleanup(ds_id, required_tags_list)

                # check if all tasks are dead 
                else:
                    dead_cnt = s.query(DebbyTask)\
                                .filter(DebbyTask.debbyscan_id == ds_id)\
                                .filter(DebbyTask.state.in_(STATES_DEAD))\
                                .count()

                    if cnt == dead_cnt:
                        s.query(DebbyScan).filter(DebbyScan.id == ds_id).update({DebbyScan.state: STATE_FINISHED})
                        if project_engine == ENGINE_MOLLY:
                            infra.stop_event(ds_id)
                        log_scan_state.s(ds_id, state=STATE_FINISHED).apply_async(countdown=1)
                        engine.scan_cleanup(ds_id, required_tags_list)

            s.query(DebbyScan).filter(DebbyScan.id == ds_id).update({DebbyScan.all_targets_tasked: True})
        safe_query(proj_run_ending)

    except Exception as e:
        print("[!] " + "-" * 39 + "[!] GLOBAL TASK ABORTING CAUSE OF EXCEPTION" + "[!] " + "-" * 39)
        traceback.print_exc()
        if ds_id:
            safe_query(lambda s:
                s.query(DebbyScan).filter(DebbyScan.id == ds_id).update({DebbyScan.state: STATE_ABORTED})
            )
            log_scan_state.s(ds_id, state=STATE_ABORTED).apply_async(countdown=1)
            if project_engine == ENGINE_MOLLY:
                infra.stop_event(ds_id)

            session = new_session()
            project = session.query(DebbyProject).filter(DebbyProject.id == project_id).first()
            project_engine = project.engine
            session.close()
            engine = new_engine(project_engine)
            engine.scan_cleanup(ds_id, required_tags_list)


@celery_app.task
def log_scan_state(scan_id, state=STATE_IN_PROGRESS):

    if state not in [STATE_IN_PROGRESS, STATE_FINISHED]:
        print('[+] log_scan_state. scan_id={}. state={}'.format(scan_id, state))

    # get info from db
    session = new_session()
    scan = session.query(DebbyScan).filter(DebbyScan.id == scan_id).first()
    if not scan:
        print('[!] log_scan_state. Not scan! scan_id: {}.'.format(scan_id))
        session.close()
        return

    project = session.query(DebbyProject).filter(DebbyProject.id == scan.project_id).first()
    required_tags = session.query(DebbyTag).filter(RelationProjectTag.project_id == project.id)\
                                           .filter(RelationProjectTag.tag_id == DebbyTag.id).all()
    policy = session.query(DebbyPolicy).filter(DebbyPolicy.id == project.policy_id).first()
    session.close()

    # state_type
    if state in STATES_ALIVE:
        state_type = STATE_TYPE_ALIVE
    elif state in STATES_DEAD:
        state_type = STATE_TYPE_DEAD
    else:
        state_type = STATE_TYPE_UNKNOWN

    # prepare event data
    event_data = {
        'state': state,
        'state_type': state_type,
        'scanId': scan.id,
        'start_time': datetime_msk_2_timestamp_utc(scan.create_time) if scan.create_time else None,
        'stop_time': datetime_msk_2_timestamp_utc(scan.finish_time) if scan.finish_time else None,
        'projectId': project.id,
        'projectName': project.name,
        'tags': list([telem.value for telem in required_tags]),
        'logClosed': project.log_closed,
        'engine': project.engine,
        'policyId': policy.id,
        'policyName': policy.name,
        'event_type': EVENT_TYPE_STATE_CHANGE
    }

    send_event(event_data)


@celery_app.task
def fetch_macros():
    fetch_golem_macros()


@celery_app.task
def solomon_routine_task():
    solomon_routine()


@celery_app.task
def solomon_daily_routine_task():
    solomon_daily_routine()


@celery_app.task
def solomon_weekly_routine_task():
    solomon_weekly_routine()


@celery_app.task
def agents_jobs_fix():
    s = new_session()

    agents = s.query(DebbyAgent.id).all()
    agents_ids = [agent.id for agent in agents]

    for agent_id in agents_ids:
        active_jobs = s.query(DebbyTask).filter(DebbyTask.state.in_(STATES_ALIVE))\
                                        .filter(DebbyTask.debbyagent_id == agent_id).count()
        s.query(DebbyAgent).filter(DebbyAgent.id == agent_id).update({'jobs': active_jobs})

    s.commit()
    s.close()


@celery_app.task
def autocancel_long_scans():
    s = new_session()

    cur_time = now()

    rows = s.query(DebbyScan.id, DebbyScan.create_time, DebbyProject.max_scan_time)\
            .filter(DebbyScan.project_id == DebbyProject.id)\
            .filter(DebbyScan.state.in_(STATES_ALIVE))\
            .all()

    for row in rows:
        project_max_scan_time = row[2]
        if project_max_scan_time is None or project_max_scan_time <= 0:
            continue

        scan_id = row[0]
        scan_create_time = row[1]

        if scan_create_time + datetime.timedelta(minutes=project_max_scan_time) < cur_time:
            
            print("[+] autocancel_long_scans. Attemp to cancel scan {}.".format(scan_id))
            
            is_canceled = cancel_scan(scan_id)
            if is_canceled:
                log_scan_state.s(scan_id, state=STATE_CANCELED).apply_async()

    s.commit()
    s.close()


@celery_app.task
def clear_old_cache_and_vacuum():
    logging.warning("[+] task 'clear_old_cache_and_vacuum' started.")

    # Delete old cache entries
    old_offset = datetime.datetime.utcnow() - datetime.timedelta(days=30)
    s = new_session()
    stmt = delete(PortCache).where(PortCache.last_seen < old_offset)
    s.execute(stmt)
    s.commit()
    s.close()

    # VACUUM
    engine = get_engine()
    connection = engine.raw_connection()
    connection.set_isolation_level(ISOLATION_LEVEL_AUTOCOMMIT)
    cursor = connection.cursor()
    cursor.execute("VACUUM {}".format(PortCache.__tablename__))
    cursor.close()
    connection.close()

    logging.warning("[+] task 'clear_old_cache_and_vacuum' finished.")

# @celery_app.on_after_configure.connect
# def setup_periodic_tasks(sender, **kwargs):
#     sender.add_periodic_task(
#         crontab(hour=0, minute=58),
#         clear_old_cache_and_vacuum.s(),
#     )
