

import json
import requests
import tenacity

try:
    from json.decoder import JSONDecodeError
except ImportError:
    JSONDecodeError = ValueError

from app.settings import HEC_HOST, HEC_TOKEN
from app.settings import FEATURE_ENABLE_SEND_RESULTS_TO_SPLUNK


def send_event(event_data, task_id=None):

    if not FEATURE_ENABLE_SEND_RESULTS_TO_SPLUNK:
        return False

    requests_url = "https://%s/services/collector" % HEC_HOST
    post_data = {
        "event": event_data
    }

    data = json.dumps(post_data).encode('utf8')
    headers = {'Authorization': "Splunk {}".format(HEC_TOKEN)}

    try:

        req = requests.post(requests_url, data=data, headers=headers, verify=False)

        response_json = json.loads(str(req.text))

        if "text" in response_json:
            if response_json["text"] != "Success":
                # print("[+] SPLUNK: Error sending request: {}".format(response_json))
                pass
            else:
                # print("[+] SPLUNK: Success. task.id: {}. retries remained: {}".format(task_id, retries))
                return True
        else:
            # print("[+] SPLUNK: Error sending request: {}".format(response_json))
            pass

    except (requests.exceptions.RequestException, ValueError) as e:
        print("[+] SPLUNK Unexpected error: {}".format(e))

    # print('SPLUNK NOT SENT')
    return False


def send_batch(events, task_id=None, scan_id=None):
    """

    :param events:
    [
      {
        "protocol": "ipv4",
        "event_type": "info",
        "tags": [
          "INTERNAL",
          "IPv4",
          "IPv6"
        ],
        "projectName": "test57",
        "logClosed": false,
        "taskId": 964,
        "scripts": [],
        "transport": "tcp",
        "dest_port": 22,
        "scanId": 411,
        "scanStartTime": 1541678338,
        "enabled": true,
        "time": 1541678339,
        "dest_ip": "5.255.255.55",
        ...
      }
    ]
    :param task_id:
    :param scan_id:
    :return:
    """

    if not FEATURE_ENABLE_SEND_RESULTS_TO_SPLUNK:
        return False

    requests_url = "https://%s/services/collector" % HEC_HOST
    post_data = [dict(event=event) for event in events]
    auth_header = "Splunk %s" % HEC_TOKEN
    headers = {'Authorization': auth_header}

    try:

        req = requests.post(requests_url, json=post_data, headers=headers, verify=False)
        response_json = req.json()

        if response_json.get("text") != "Success":
            # print('[+] SPLUNK: Error sending data: {}. task_id: {}. scan_id: {}.'.format(
            #       response_json, task_id, scan_id))
            pass
        else:
            # print('[+] SPLUNK: Success. task_id: {}'.format(task_id))
            return True

    except (JSONDecodeError, requests.exceptions.RequestException) as e:
        print("[+] SPLUNK Unexpected error: {}. task_id: {}. scan_id: {}.".format(e, task_id, scan_id))

    # print('SPLUNK NOT SENT')
    return False


@tenacity.retry(retry=tenacity.retry_if_result(lambda res: not res), wait=tenacity.wait_fixed(5),
                stop=tenacity.stop_after_attempt(5), retry_error_callback=(lambda rstate: False))
def send_log_to_splunk(splunk_events, task_id, scan_id):

    if not splunk_events or not FEATURE_ENABLE_SEND_RESULTS_TO_SPLUNK:
        return False

    res = send_batch(splunk_events, task_id, scan_id)
    return res
