

import calendar
import datetime
import functools
import ipaddress
import pytz
import requests
import socket
import sys
import tenacity

from app.settings import PROTO_IPV4, PROTO_IPV6


def ipaddress_check(func):
    """Convert input value according to python version. And return False if not valid stringlike type."""
    @functools.wraps(func)
    def wrapper(value_):

        # decoding and checking "value" type for py2 and py3 separately.
        if sys.version_info[0] < 3:
            # python2

            if isinstance(value_, str):
                value = value_.decode()
            else:
                value = value_

            if not isinstance(value, str):
                return False

        else:
            # python3

            value = value_
            if not isinstance(value, str):
                return False

        # orig func
        res = func(value)
        return res

    return wrapper


# def is_subnet_DEPRICATED(value):
#     parts = value.split('/')
#     if len(parts) != 2:
#         return False

#     try:
#         ipaddress.IPv4Address(parts[0])
#         return parts[1].isdigit() and 0 <= int(parts[1]) <= 32
#     except ipaddress.AddressValueError:
#         pass

#     try:
#         ipaddress.IPv6Address(parts[0])
#         return parts[1].isdigit() and 0 <= int(parts[1]) <= 128
#     except ipaddress.AddressValueError:
#         pass

#     return False


@ipaddress_check
def is_subnet(value):
    """
    is_subnet("192.168.1.1") -> False
    is_subnet("192.168.1.1/30") -> True
    is_subnet("192.168.1.1/5") -> True
    is_subnet([]) -> False
    is_subnet("0") -> False
    is_subnet(1) -> False
    """

    # check ip network (also accepts ip addrs like "192.168.1.1")
    try:
        ipaddress.ip_network(value, strict=False)
    except ValueError:
        return False

    # Check presence of symbol "/",
    #   because ipaddress.ip_network accepts ip addrs like "192.168.1.1".
    if "/" not in value:
        return False

    return True


# def is_valid_ipv4_address_DEPRICATED(address):
#     try:
#         socket.inet_pton(socket.AF_INET, address)
#     except AttributeError:  # no inet_pton here, sorry
#         try:
#             socket.inet_aton(address)
#         except socket.error:
#             return False
#         return address.count('.') == 3
#     except socket.error:  # not a valid address
#         return False
#     return True


@ipaddress_check
def is_valid_ipv4_address(address):
    try:
        ipaddress.IPv4Address(address)
        return True
    except ValueError:
        return False


def is_valid_ipv6_address(address):
    try:
        socket.inet_pton(socket.AF_INET6, address)
    except socket.error:  # not a valid address
        return False
    return True


def extract_target_list_from_string(input_):
    """
    Input:
        "5.255.255.5, 5.255.255.80,5.255.255.88, , ::1,   yandex.ru"
    Output:
        ["5.255.255.5", "5.255.255.80", "5.255.255.88", "::1", "yandex.ru"]
    """

    input_ = input_.replace(' ', '')
    return list([x for x in input_.split(',') if x])


# --------------------------------
# --- FIXING TIMEZONE PROBLEMS ---
# --------------------------------


# def datetime_2_timestamp(dt):
#     return calendar.timegm(dt.timetuple())
#
#
# def timestamp_2_datetime(ts):
#     return datetime.datetime.fromtimestamp(ts)


def datetime_msk_2_timestamp_utc(dt):
    utc_now = datetime.datetime.utcnow().replace(microsecond=0)
    msk_now = utc_dt_to_msk_dt(utc_now)
    delta = msk_now - utc_now
    dt = dt - delta
    return calendar.timegm(dt.timetuple())


def timestamp_utc_2_datetime_msk(ts):
    return utc_dt_to_msk_dt(datetime.datetime.utcfromtimestamp(ts))


def utc_dt_to_msk_dt(utc_dt):
    return utc_dt.replace(tzinfo=pytz.utc).astimezone(pytz.timezone('Europe/Moscow')).replace(tzinfo=None)


# def msk_dt_to_utc_dt(utc_dt):
#     return utc_dt.replace(tzinfo=pytz.timezone('Europe/Moscow')).astimezone(pytz.utc).replace(tzinfo=None)


def now():
    utc_dt = datetime.datetime.utcnow().replace(microsecond=0)
    return utc_dt_to_msk_dt(utc_dt)


# ----------------------
# --- Ports iterator ---
# ----------------------


def iterate_ports(ports_str):
    """ "1, 2,3, 6-8"   =>   [1, 2, 3, 6, 7, 8] """

    ports_str = ports_str.replace(' ', '')
    ports_list = ports_str.replace(' ', '').split(',')
    for descriptor in ports_list:
        pp = descriptor.split('-')
        if len(pp) == 1:
            yield int(pp[0])
        else:
            for p in range(int(pp[0]), int(pp[1]) + 1):
                yield p


def uniq_ports(ports_list):
    """
    Input list of ports ids
    Return not sorted but unique list

    :param ports_list: [1, 22, 3, 11, 22]
    :return: [11, 1, 3, 22]
    """

    return list(set(ports_list))


def port_list_to_ports_str(port_list):
    """
    Input list of ports
    Return string of ports separated by ","

    :param port_list: [1, 2, 5, 11, 2]
    :return: "1,2,5,11,2"
    """

    return ",".join(str(x) for x in port_list)

# -----------------------
# --- ipaddress utils ---
# -----------------------


def addr_to_proto(addr):
    """
    :param addr:
    :return: PROTO_IPV4 or PROTO_IPV6
    """
    try:
        ipaddress.IPv4Address(addr)
    except ipaddress.AddressValueError:
        return PROTO_IPV6
        # try:
        #     ipaddress.IPv4Address(addr.decode())
        # except ipaddress.AddressValueError:
        #     return PROTO_IPV6
    return PROTO_IPV4


def str_to_ipaddress_any(addr_str):
    """
    :param addr_str:
    :return: ipaddress.IPv6Address object or ipaddress.IPv4Address object or None
    """
    try:
        return ipaddress.ip_address(addr_str)
    except ipaddress.AddressValueError:
        pass
        # try:
        #     return ipaddress.ip_address(addr_str.decode())
        # except ipaddress.AddressValueError:
        #     pass

    return None


def str_to_ipaddress6(addr_str):
    """
    :param addr_str:
    :return: ipaddress.IPv6Address object or None
    """
    try:
        return ipaddress.IPv6Address(addr_str)
    except ipaddress.AddressValueError:
        pass
        # try:
        #     return ipaddress.IPv6Address(addr_str.decode())
        # except ipaddress.AddressValueError:
        #     pass

    return None


def str_to_ipaddress4(addr_str):
    """
    :param addr_str:
    :return: ipaddress.IPv4Address object or None
    """
    try:
        return ipaddress.IPv4Address(addr_str)
    except ipaddress.AddressValueError:
        pass
        # try:
        #     return ipaddress.IPv4Address(addr_str.decode())
        # except ipaddress.AddressValueError:
        #     pass

    return None


def str_to_ipnetwork_any(net_str):
    """
    :param net_str:
    :return: ipaddress.IPv4Address object or None
    """
    try:
        return ipaddress.ip_network(net_str)
    except ipaddress.AddressValueError:
        pass
        # try:
        #     return ipaddress.ip_network(net_str.decode())
        # except (ipaddress.AddressValueError, ValueError):
        #     pass

    return None


def str_to_ipnetwork4(net_str):
    """
    :param net_str:
    :return: ipaddress.IPv4network object or None
    """
    try:
        return ipaddress.IPv4Network(net_str)
    except ipaddress.AddressValueError:
        pass
        # try:
        #     return ipaddress.IPv4Network(net_str.decode())
        # except (ipaddress.AddressValueError, ValueError):
        #     pass

    return None


def str_to_ipnetwork6(net_str):
    """
    :param net_str:
    :return: ipaddress.IPv6network object or None
    """
    try:
        return ipaddress.IPv6Network(net_str)
    except ipaddress.AddressValueError:
        pass
        # try:
        #     return ipaddress.IPv6Network(net_str.decode())
        # except (ipaddress.AddressValueError, ValueError):
        #     pass

    return None


# -------------------
# --- build event ---
# -------------------

def build_splunk_event(**kwargs):
    all_kwargs = [
        'event_type',
        'projectId', 'projectName', 'engine', 'logClosed', 'tags',
        'policyId',
        'dest_ip', 'dest_host', 'resp', 'resp_source', 'protocol',
        'dest_port', 'transport', 'time', 'enabled', 'scripts',
        'service_name', 'service_product', 'service_version',
        'scanId', 'scanStartTime',
        'taskId'
    ]

    specified_kwargs = kwargs
    for key in all_kwargs:
        if key not in list(specified_kwargs.keys()):
            specified_kwargs.update({key: None})

    return specified_kwargs


# ------------------
# --- Get owners ---
# ------------------

# --- GOLEM ---
# DEPRICATED
# DEPRICATED
# DEPRICATED

# @tenacity.retry(retry=tenacity.retry_if_exception_type(requests.exceptions.ConnectionError),
#                 wait=tenacity.wait_fixed(2), stop=tenacity.stop_after_attempt(10), reraise=True)
# def golem_get_resp_by_fqdn(fqdn):
#     response = requests.get('http://ro.admin.yandex-team.ru/api/get_host_resp.sbml', params={'hostname': fqdn})
#     if response.status_code != 200 or not response.text:
#         return []
#     return list(filter(lambda x: x, map(lambda x: x.strip(), filter(lambda x: x, response.text.split(',')))))


# --- APPSEC-DISCOERY ---


class AppSecDiscoveryClient(object):

    @staticmethod
    @tenacity.retry(wait=tenacity.wait_random_exponential(max=10), stop=tenacity.stop_after_attempt(5))
    def _get_resp(json_data):
        url = 'https://appsec-discovery.sec.yandex-team.ru/api/for_services/search_host_owner'
        resp = requests.post(url, json=json_data, verify=False, timeout=3).json()
        results = resp.get('results') 
        if results:
            return results.get('owners')
            # return results.get('owners'), results.get('source')
        else:
            return list()

    @staticmethod
    def get_resp_by_ip(ip):
        return AppSecDiscoveryClient._get_resp({'ip': ip})

    @staticmethod
    def get_resp_by_fqdn(fqdn):
        return AppSecDiscoveryClient._get_resp({'fqdn': fqdn})



# --- OAUTH ---

# def get_safe_xml_parser():
#     class FakeDTDResolver(etree.Resolver):
#         def resolve(self, url, id, context):
#             return self.resolve_string('', context)
#     parser = etree.XMLParser(remove_blank_text=True, resolve_entities=False, no_network=True)
#     parser.resolvers.add(FakeDTDResolver())
#     return parser


# def oauth_to_uid(oauth_token, userip='127.0.0.1'):
#
#     try:
#         url = 'https://blackbox.yandex-team.ru/blackbox'
#         scopes = ','.join(['debby:init', 'debby:read'])
#         params = {
#             'method': 'oauth',
#             'userip': userip,
#             'scopes': scopes
#         }
#         headers = {'Authorization': 'OAuth {}'.format(oauth_token)}
#         r = requests.get(url, params=params, headers=headers)
#
#         parser = get_safe_xml_parser()
#         xml = etree.fromstring(bytes(r.text), parser)
#
#         error = xml.find('error')
#         if error is None or error.text != 'OK':
#             return None
#
#         uid = xml.find('uid')
#         if uid is None:
#             return None
#         else:
#             return int(uid.text)
#
#     except:
#         traceback.print_exc()
#         print('[!] oauth_to_uid. Exception occured.')
#
#     return None

# --- ERROR PRINT ---

def eprint(*args, **kwargs):
    print(*args, file=sys.stderr, **kwargs)
