
import re
import datetime
# import logging

from app.settings import ENGINES
from app.utils import timestamp_utc_2_datetime_msk
from app.settings import TRANSPORT_PROTOCOLS, AVAILABLE_NMAP_SCRIPTS


class DebbyException(Exception):
    pass


class DebbyValidateException(DebbyException):
    pass


class DebbyRuntimeException(DebbyException):
    pass


class DebbyOwnerException(DebbyException):
    pass


def prepare_project_targets(targets_string):
    """
    targets_string: "A,B, C,D exclude D, E"
    return: "A, B, C, D exclude D, E"
    """

    parts = targets_string.split(' exclude ')

    if len(parts) == 2 and parts[0][-1] != ',' and parts[1][0] != ',':
        left = parts[0].replace(' ', '').split(',')
        right = parts[1].replace(' ', '').split(',')
        prepared = ', '.join(left) + ' exclude ' + ', '.join(right)
    else:
        prepared = ', '.join(targets_string.replace(' ', '').split(','))

    return prepared


def validate_namespace(request):
    # NAME
    name = request.form.get("namespace_name")
    if len(name) == 0 or len(name) > 128:
        raise DebbyValidateException('Namespace name should have > 0 and <= 128 symbols')

    # CONTACTS
    contacts = request.form.get("namespace_contacts")
    contacts = contacts if contacts else ''
    if contacts and not re.match(r"^[\w\d\-_, ]*$", contacts):
        raise DebbyValidateException('Incorrect contacts value. Must match regex: "^[\w\d-_]*$".')
    
    return {
        "name": name,
        "contacts": contacts,
    }


def validate_project(request, known_tags, known_policies, known_projects, known_namespaces):
    # NAME
    name = request.form.get("project_name")
    if len(name) == 0 or len(name) > 128:
        raise DebbyValidateException('Project name should have > 0 and <= 128 symbols')

    # TARGETS: IPv4, IPv6, domain name, macros
    targets = request.form.get("project_targets").strip()
    target_list = prepare_project_targets(targets)
    if len(target_list) == 0:
        raise DebbyValidateException('No targets were specified')

    # ENGINE
    engine = request.form.get("project_engine")
    if engine not in ENGINES:
        raise DebbyValidateException('Unknown engine %s' % engine)

    # START AND STOP TIME
    scan_start = request.form.get("project_start")
    scan_stop = request.form.get("project_stop")
    try:
        dt_start = datetime.datetime.strptime(scan_start, "%d.%m.%Y %H:%M")
        dt_stop = datetime.datetime.strptime(scan_stop, "%d.%m.%Y %H:%M")
        if dt_start > dt_stop:
            raise ValueError
    except ValueError:
        raise DebbyValidateException('Incorrect data format')

    # INTERVAL
    scan_interval = request.form.get("project_interval")
    try:
        if int(scan_interval) < 0:
            raise ValueError
        ts_interval = timestamp_utc_2_datetime_msk(int(scan_interval))
    except ValueError:
        raise DebbyValidateException('Incorrect interval format')

    # INTERVAL
    max_scan_time = request.form.get("max_scan_time")
    try:
        if not max_scan_time.isdigit() and int(max_scan_time) < 0:
            raise ValueError
    except ValueError:
        raise DebbyValidateException('Incorrect max_scan_time format')

    # POLICY
    policy_name = request.form.get("project_policy")
    try:
        policy_id = int(policy_name.split(':')[0])
        if policy_id not in list([p.id for p in known_policies]):
            raise DebbyValidateException('Unknown policy specified')
    except ValueError:
        raise DebbyValidateException('Incorrect policy specified')

    # Namespace
    namespace = request.form.get("project_namespace")
    try:
        namespace_id = int(namespace.split(':')[0])
        if namespace_id not in list([ns.id for ns in known_namespaces]):
            raise DebbyValidateException('Unknown namespace specified')
    except ValueError:
        raise DebbyValidateException('Incorrect namespace specified')
    if namespace_id == 0:
        namespace_id = None

    # TAGS
    spec_tags = request.form.getlist("project_tags")
    # logging.warning('Watch out!')
    # logging.warning('ERGSJH4TUW45')
    # logging.warning(spec_tags)
    spec_tags_ids = list([int(t.split(':')[0]) for t in spec_tags])
    # logging.warning(spec_tags_ids)
    # logging.warning('ERGSJH4TUW45')
    if list(sorted(spec_tags_ids)) != list(sorted(list(set(spec_tags_ids)))):
        raise DebbyValidateException('Incorrect tag specified1')

    known_tags_ids = list([t.id for t in known_tags])
    if not all([spec_tag_id in known_tags_ids for spec_tag_id in spec_tags_ids]):
        raise DebbyValidateException('Incorrect tag specified2')

    # PIPELINE PROJECTS
    spec_pipeline_projs = request.form.getlist("pipeline_projects")
    spec_pipeline_projs_ids = list([int(p.split(':')[0]) for p in spec_pipeline_projs])
    if list(sorted(spec_pipeline_projs_ids)) != list(sorted(list(set(spec_pipeline_projs_ids)))):
        raise DebbyValidateException('Incorrect pipeline project specified1')

    known_projects_ids = list([t.id for t in known_projects])
    if not all([ppid in known_projects_ids for ppid in spec_pipeline_projs_ids]):
        raise DebbyValidateException('Incorrect pipeline project specified2')

    # OPTIONS
    log_closed = request.form.get("project_log_closed")
    save_to_db = request.form.get("project_save_to_db")
    log_closed = True if log_closed else False
    save_to_db = True if save_to_db else False

    # ST TICKET
    st_ticket = request.form.get("project_st_ticket")
    st_ticket = st_ticket if st_ticket else None
    if st_ticket and not re.match(r"^[\w\d]+-\d+$", st_ticket):
        raise DebbyValidateException('Invalid st ticket')

    # RETRIES
    retries = request.form.get("project_retries")
    retries = retries if retries else "0"
    try:
        if not retries.isdigit() and int(retries) < 0:
            raise ValueError
    except ValueError:
        raise DebbyValidateException('Incorrect retries value. Must be positive integer.')

    # RETRY PERIOD
    retry_period = request.form.get("project_retry_period")
    retry_period = retry_period if retry_period else "0"
    try:
        if not retry_period.isdigit() and int(retry_period) < 0:
            raise ValueError
    except ValueError:
        raise DebbyValidateException('Incorrect retry_period value. Must be positive integer.')

    # CONTACTS
    contacts = request.form.get("project_contacts")
    contacts = contacts if contacts else ''
    if contacts and not re.match(r"^[\w\d\-_, ]*$", contacts):
        raise DebbyValidateException('Incorrect contacts value. Must match regex: "^[\w\d-_]*$".')

    return {
        'name': name,
        'target_list': target_list,
        'engine': engine,
        'dt_start': dt_start,
        'dt_stop': dt_stop,
        'ts_interval': ts_interval,
        'max_scan_time': max_scan_time,
        'spec_tags_ids': spec_tags_ids,
        'policy_id': policy_id,
        'log_closed': log_closed,
        'save_to_db': save_to_db,
        'spec_pipeline_projs_ids': spec_pipeline_projs_ids,
        'st_ticket': st_ticket,
        'retries': retries,
        'retry_period': retry_period,
        'contacts': contacts,
        'namespace_id': namespace_id,
    }


def validate_new_policy(request):
    # NAME
    name = request.form.get("policy_name")
    if len(name) == 0 or len(name) > 32:
        raise DebbyValidateException('Policy name length: more than 0 and not more than 32 symbols')

    # PORTS
    ports = request.form.get("policy_ports")
    if ports:
        ports_list = ports.replace(' ', '').split(',')
        if not all([check_port(p) for p in ports_list]):
            raise DebbyValidateException('Incorrect ports')
    else:
        ports_list = []

    # PROTO
    scan_type = request.form.get("policy_scan_type")
    if scan_type not in TRANSPORT_PROTOCOLS:
        raise DebbyValidateException('Unknown scan type')

    # SCRIPTS
    spec_scripts = request.form.getlist("policy_scripts")
    for script_name in spec_scripts:
        if script_name not in AVAILABLE_NMAP_SCRIPTS:
            raise DebbyValidateException('Unknown script specified')

    # ADDITIONAL OPTIONS
    additional_options = request.form.get("additional_options")
    # TODO: Fix this strange filter
    for c in ['|', '&', '>', '<']:
        if c in additional_options:
            raise DebbyValidateException("Prohibited symbol '{}'".format(c))

    res = {
        'name': name,
        'ports_list': ports_list,
        'scan_type': scan_type,
        'spec_scripts': spec_scripts,
        'additional_options': additional_options
    }

    return res


def check_port(port_str):
    # 50
    if port_str.isdigit():
        return True
    # 1-100
    splitted = port_str.split('-')
    if (
            (len(splitted) == 2) and (splitted[0].isdigit()) and
            (splitted[1].isdigit()) and (int(splitted[0]) < int(splitted[1]))
    ):
        return True

    return False
