import functools
import flask
import json
import logging

import blackbox
from app.settings import ADMINS, DEBBY_SKIP_YAUTH, DEBBY_SITE, DONT_CHECK_API_AUTH
from app.tvm import TvmClient, get_blackbox_tvm_ticket


def admin_role_required(func):

    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        if DEBBY_SKIP_YAUTH:
            return func(*args, **kwargs)

        if flask.g.yalogin not in ADMINS:
            flask.abort(403)

        return func(*args, **kwargs)

    return wrapper


def sessionid_required(func):

    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        if DEBBY_SKIP_YAUTH:
            return func(*args, **kwargs)

        validator = _SessionidValidator()
        blackbox_response = validator.validate_sessionid_from_request()

        if not validator.blackbox_response_is_valid(blackbox_response):
            return flask.redirect(validator.get_passport_auth_url())
        
        flask.g.yalogin = validator.login_from(blackbox_response)
        flask.g.yauid = validator.uid_from(blackbox_response)
        # flask.g.user_ticket = validator.user_ticket(blackbox_response)
        return func(*args, **kwargs)

    return wrapper


class _SessionidValidator(object):
    userip_variable_name = 'HTTP_X_REAL_IP'
    # passport_auth_url = 'https://passport.yandex-team.ru/auth?retpath=%s'

    def __init__(self):
        self.blackbox = blackbox.JsonBlackbox(url='https://blackbox.yandex-team.ru/blackbox')

    def validate_sessionid_from_request(self):
        if self._sessionid_not_set():
            return None
        try:
            response = self.blackbox.sessionid(
                userip=self._get_userip(),
                sessionid=self._get_sessionid(),
                host=flask.request.host,
                sslsessionid=self._get_sslsessionid(),
                # get_user_ticket="yes",
                headers={'X-Ya-Service-Ticket': get_blackbox_tvm_ticket()}
            )
        except blackbox.BlackboxResponseError as exc:
            # from scenter import app
            # sentry_logger = logging.getLogger(app.config['SENTRY_LOGGER'])
            # sentry_logger.error('Blackbox error: %s', exc)
            return None
        else:
            return response

    def _sessionid_not_set(self):
        return not(self._get_sessionid() and self._get_sslsessionid())

    def _get_sessionid(self):
        return flask.request.cookies.get('Session_id')

    def _get_host(self):
        return flask.request.headers.get('Host')

    def _get_sslsessionid(self):
        return flask.request.cookies.get('sessionid2')

    def _get_userip(self):
        return flask.request.environ.get(self.userip_variable_name, 'unknown')

    def blackbox_response_is_valid(self, blackbox_response):
        return not (blackbox_response is None
                    or self._error_in(blackbox_response)
                    or self._invalid_status(blackbox_response))

    def _error_in(self, response):
        return 'exception' in response

    def _invalid_status(self, response):
        status_value = response['status']['value']
        return status_value != 'VALID'

    def login_from(self, response):
        return response['login']

    def uid_from(self, response):
        return int(response['uid']['value'])

    def user_ticket(self, response):
        return response['user_ticket']

    def get_passport_auth_url(self):
        scheme = "https://" if flask.request.is_secure else "http://"
        host_and_port = DEBBY_SITE or self._get_host()
        passport_auth_url = "https://passport.yandex-team.ru/auth?retpath="
        return passport_auth_url + scheme + host_and_port
        # if DEBBY_SITE:
        #     return self.passport_auth_url % DEBBY_SITE
        # else:
        #     return self.passport_auth_url % self._get_host()
        # return self.passport_auth_url


def user_group_required(usernames):
    def _wrapper(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):

            if (not DONT_CHECK_API_AUTH) and (flask.g.yalogin not in usernames):
                flask.abort(403)
            return func(*args, **kwargs)

        return wrapper
    return _wrapper


def oauth_scope_required(required_scopes):
    def _wrapper(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):

            if DONT_CHECK_API_AUTH:
                flask.g.yalogin = ''
                flask.g.yauid = 0
                flask.g.yascope = []
                # flask.g.user_ticket = ''
                return func(*args, **kwargs)

            else:
                validator = _OauthValidator()

                blackbox_response = validator.validate_oauth_from_request()

                if not validator.blackbox_response_is_valid(blackbox_response):
                    return flask.abort(403)

                flask.g.yalogin = validator.login_from(blackbox_response)
                flask.g.yauid = validator.uid_from(blackbox_response)
                flask.g.yascope = validator.scope_from(blackbox_response)
                # flask.g.user_ticket = validator.user_ticket(blackbox_response)

                # if list(set(flask.g.yascope) & set(required_scopes)):
                #     return func(*args, **kwargs)

                if set(required_scopes).issubset(set(flask.g.yascope)):
                    return func(*args, **kwargs)

                return flask.abort(403)

        return wrapper
    return _wrapper


class _OauthValidator(object):
    userip_variable_name = 'HTTP_X_REAL_IP'

    def __init__(self):
        self.blackbox = blackbox.JsonBlackbox(url='https://blackbox.yandex-team.ru/blackbox')

    def validate_oauth_from_request(self):
        try:
            response = self.blackbox.oauth(
                oauth_token=self._get_oauth(),
                userip=self._get_userip(),
                by_token=True,
                host=flask.request.host,
                # get_user_ticket="yes",
                headers={'X-Ya-Service-Ticket': get_blackbox_tvm_ticket()}
            )
        except blackbox.BlackboxResponseError as exc:
            # from scenter import app
            # sentry_logger = logging.getLogger(app.config['SENTRY_LOGGER'])
            # sentry_logger.error('Blackbox error: %s', exc)
            return None
        else:
            return response

    def _get_oauth(self):
        auth = flask.request.headers.get('Authorization', '').split()
        if not auth or auth[0].lower() != b'oauth' or len(auth) != 2:
            return flask.abort(401)
        return auth[-1]

    def _get_userip(self):
        return flask.request.environ.get(self.userip_variable_name, 'unknown')

    def blackbox_response_is_valid(self, blackbox_response):
        return not (blackbox_response is None
                    or self._error_in(blackbox_response)
                    or self._invalid_status(blackbox_response))

    def _error_in(self, response):
        return 'exception' in response

    def _invalid_status(self, response):
        status_value = response['status']['value']
        return status_value != 'VALID'

    def login_from(self, response):
        return response['login']

    def uid_from(self, response):
        return int(response['uid']['value'])

    def scope_from(self, response):
        return response['oauth']['scope'].split()

    def user_ticket(self, response):
        return response['user_ticket']

#########

ROLES_ADMIN = "/role/admin/"
ROLES_PRIVATE_API = "/role/private-api/"
ROLES_PUBLIC_API = "/role/public-api/"

def _check_role_existance(uid, role):
    roles = json.loads(TvmClient().get().get_roles().raw)
    # logging.warning("roles: {}".format(roles))

    users_roles = roles.get("user")
    # logging.warning("users_roles: {}".format(users_roles))
    if not users_roles:
        return False
    
    user_roles = users_roles.get(str(uid))
    logging.warning("_check_role_existance. uid: {}. user_roles: {}".format(uid, user_roles))
    if not user_roles:
        return False

    # logging.warning("user_roles.keys(): {}".format(user_roles.keys()))
    return role in user_roles.keys()

# def require_role(role):
#     def decorator(function):
#         def wrapper(*args, **kwargs):
#             if not _check_role_existance(flask.g.yauid):
#                 flask.abort(403)
#             return function(*args, **kwargs)
#         return wrapper
#     return decorator

def require_role(role):
    def decorator(function):
        @functools.wraps(function)
        def wrapper(*args, **kwargs):
            if DEBBY_SKIP_YAUTH:
                return function(*args, **kwargs)

            res = _check_role_existance(flask.g.yauid, role)
            logging.warning("check role {} for uid {}, res: {}".format(role, flask.g.yauid, res))
            if not res:
                flask.abort(403)

            return function(*args, **kwargs)
        return wrapper
    return decorator