import flask
from flask import Flask, request, jsonify
import functools
from base64 import b64encode, b64decode
# import logging
# logging.basicConfig(filename='debug.log', level=logging.DEBUG)
from app.openvas_client import OpenVASClient, OpenVASClientException


app = Flask(__name__)


def get_username_password_from_headers(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):

        openvas_username = request.headers.get('X-OPENVAS-USERNAME', 'admin')
        openvas_password = request.headers.get('X-OPENVAS-PASSWORD')

        if not openvas_password:
            flask.abort(403)
        else:
            flask.g.openvas_username = openvas_username
            flask.g.openvas_password = openvas_password

        return func(*args, **kwargs)

    return wrapper


def prepare_openvas_client(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):

        if not flask.g.openvas_password:
            flask.abort(403)
        else:
            cli = OpenVASClient()
            cli.set_password(flask.g.openvas_password)
            if flask.g.openvas_username:
                cli.set_username(flask.g.openvas_username)

            flask.g.openvas_client = cli

        return func(*args, **kwargs)

    return wrapper


@app.route('/task/<task_uuid>', methods=['POST'])
@get_username_password_from_headers
@prepare_openvas_client
def run_task(task_uuid):

    report_uuid = flask.g.openvas_client.run_task(task_uuid)
    
    if report_uuid:
        return jsonify({"status": "ok", "res": {"report_uuid": report_uuid}})
    else:
        return jsonify({"status": "error"})


@app.route('/task/<task_uuid>', methods=['DELETE'])
@get_username_password_from_headers
@prepare_openvas_client
def stop_task(task_uuid):

    res = flask.g.openvas_client.stop_task(task_uuid)
    
    if res:
        return jsonify({"status": "ok"})
    else:
        return jsonify({"status": "error"})


@app.route('/report/<report_uuid>', methods=['GET'])
@get_username_password_from_headers
@prepare_openvas_client
def get_report_info(report_uuid):

    report_info = flask.g.openvas_client.get_report_info(report_uuid)
    
    if report_info:
        return jsonify({"status": "ok", "res": report_info})
    else:
        return jsonify({"status": "error"})


@app.route('/configs/<config_uuid>', methods=['GET'])
@get_username_password_from_headers
@prepare_openvas_client
def export_config(config_uuid):

    try:
        config_str = flask.g.openvas_client.export_config(config_uuid)
        if not config_str:
            raise OpenVASClientException("Not Found")
        
        return jsonify({"status": "ok", "res": b64encode(config_str).decode()})

    except OpenVASClientException as e:
        return jsonify({"status": "error", "message": e.args[0]})


@app.route('/configs', methods=['POST'])
@get_username_password_from_headers
@prepare_openvas_client
def import_config():

    try:
        config_uuid = flask.g.openvas_client.import_config(b64decode(request.data))
        if not config_uuid:
            raise OpenVASClientException("Not Imported")
        
        return jsonify({"status": "ok", "res": config_uuid})

    except OpenVASClientException as e:
        return jsonify({"status": "error", "message": e.args[0]})


@app.route('/port_list', methods=['POST'])
@get_username_password_from_headers
@prepare_openvas_client
def create_port_list():
    port_list_uuid = flask.g.openvas_client.create_port_list(request.data)
    if not port_list_uuid:
        return jsonify({"status": "error", "message": "Was not created"})
    return jsonify({"status": "ok", "res": port_list_uuid})


@app.route('/targets', methods=['POST'])
@get_username_password_from_headers
@prepare_openvas_client
def create_target():

    try:
        targets = request.form.get('targets').split(',')
        if not targets:
            raise OpenVASClientException("targets not specified")

        port_list_uuid = request.form.get('port_list_uuid')
        # if not port_list_uuid:
        #     raise OpenVASClientException("port_list_uuid not specified")

        target_uuid = flask.g.openvas_client.create_target(targets, port_list_uuid)
        if not target_uuid:
            raise OpenVASClientException("Not created")
        
        return jsonify({"status": "ok", "res": target_uuid})

    except OpenVASClientException as e:
        return jsonify({"status": "error", "message": e.args[0]})


@app.route('/tasks', methods=['POST'])
@get_username_password_from_headers
@prepare_openvas_client
def start_task():

    try:
        config_uuid = request.form.get('config_uuid')
        if not config_uuid:
            raise OpenVASClientException("config_uuid not specified")

        target_uuid = request.form.get('target_uuid')
        if not target_uuid:
            raise OpenVASClientException("target_uuid not specified")

        scanner_uuid = flask.g.openvas_client.get_openvas_default_scanner()
        task_uuid = flask.g.openvas_client.create_task(config_uuid, target_uuid, scanner_uuid)
        if not task_uuid:
            raise OpenVASClientException("Not created")
        
        return jsonify({"status": "ok", "res": task_uuid})

    except OpenVASClientException as e:
        return jsonify({"status": "error", "message": e.args[0]})


@app.route('/tasks/<task_uuid>', methods=['DELETE'])
@get_username_password_from_headers
@prepare_openvas_client
def delete_task(task_uuid):

    ok = flask.g.openvas_client.delete_task(task_uuid)
    if not ok:
        return jsonify({"status": "error", "message": "Not Deleted"})
    
    return jsonify({"status": "ok"})


@app.route('/targets/<target_uuid>', methods=['DELETE'])
@get_username_password_from_headers
@prepare_openvas_client
def delete_target(target_uuid):

    ok = flask.g.openvas_client.delete_target(target_uuid)
    if not ok:
        return jsonify({"status": "error", "message": "Not Deleted"})
    
    return jsonify({"status": "ok"})


@app.route('/port_list/<port_list_uuid>', methods=['DELETE'])
@get_username_password_from_headers
@prepare_openvas_client
def delete_port_list(port_list_uuid):

    ok = flask.g.openvas_client.delete_port_list(port_list_uuid)
    if not ok:
        return jsonify({"status": "error", "message": "Not Deleted"})
    
    return jsonify({"status": "ok"})


@app.route('/configs/<config_uuid>', methods=['DELETE'])
@get_username_password_from_headers
@prepare_openvas_client
def delete_config(config_uuid):

    ok = flask.g.openvas_client.delete_config(config_uuid)
    if not ok:
        return jsonify({"status": "error", "message": "Not Deleted"})
    
    return jsonify({"status": "ok"})



@app.route('/ping', methods=['GET'])
def ping():
    return "PONG\n"
