package app

import (
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/debby/targets/internal/auth"
	"a.yandex-team.ru/security/debby/targets/internal/config"
	"a.yandex-team.ru/security/debby/targets/internal/cvs"
	// "a.yandex-team.ru/security/debby/targets/internal/l3mgr"
	"a.yandex-team.ru/security/debby/targets/internal/macros"
	"a.yandex-team.ru/security/debby/targets/internal/netmap"
	"a.yandex-team.ru/security/debby/targets/internal/yp"
)

type App struct {
	cfg      *config.Config
	cvsCache *cvs.CvsCache
	// l3mgrCache  *l3mgr.L3mgrCache
	logger      log.Logger
	macrosCache *macros.MacrosCache
	netmapCache *netmap.NetmapCache
	http        *echo.Echo
	ypCache     *yp.YpCache
}

func NewApp(logger log.Logger, cfg *config.Config) (*App, error) {
	// // l3mgr client and cache
	// l3mgrClient, err := l3mgr.NewClient(cfg.L3mgrToken, l3mgr.WithLogger(logger))
	// if err != nil {
	// 	return nil, fmt.Errorf("failed to create l3mgrClient: %s", err)
	// }
	// l3mgrCache := l3mgr.NewCache(l3mgrClient, l3mgr.WithCacheLogger(logger))

	// create yp client and cache
	ypCli := yp.NewYpClient(yp.WithLogger(logger))
	ypCache := yp.NewYpCache(ypCli, yp.CacheWithLogger(logger))

	// create netmap client and cache
	nc, err := netmap.NewNetmapClient(netmap.WithLogger(logger))
	if err != nil {
		return nil, fmt.Errorf("failed to create netmapClient: %s", err)
	}
	netmapCache := netmap.NewNetmapCache(nc, netmap.CacheWithLogger(logger))

	// create macros client and cache
	mc, err := macros.NewMacrosClient(macros.WithLogger(logger))
	if err != nil {
		return nil, fmt.Errorf("failed to create macrosClient: %s", err)
	}
	macrosCache := macros.NewMacrosCache(mc, macros.CacheWithLogger(logger))

	// create cvs client and cache for dnscache
	cvsOptions := []cvs.Option{
		cvs.WithLogger(logger),
		cvs.WithUsername(cfg.Cvs.Username),
		cvs.WithKeyPath(cfg.Cvs.KeyPath),
	}
	cc := cvs.NewCvsClient(cvsOptions...)
	cvsCache := cvs.NewCvsCache(cc, cvs.CacheWithLogger(logger))

	// create App
	app := App{
		cfg:      cfg,
		cvsCache: cvsCache,
		// l3mgrCache:  l3mgrCache,
		logger:      logger,
		macrosCache: macrosCache,
		netmapCache: netmapCache,
		http:        echo.New(),
		ypCache:     ypCache,
	}

	return &app, nil
}

func (app App) Initialize() {
	if !app.netmapCache.IsReady() {
		app.netmapCache.UpdateCache()
	}
	if !app.macrosCache.IsReady() {
		app.macrosCache.UpdateCache()
	}
	if !app.cvsCache.IsReady() {
		app.cvsCache.UpdateCache()
	}
	if !app.ypCache.IsReady() {
		app.ypCache.UpdateCache()
	}

	// if !app.l3mgrCache.IsReady() {
	// 	app.l3mgrCache.UpdateCache()
	// }

	// initialize http endpoints
	app.http.Use(middleware.Recover())
	app.http.GET("/ping", func(c echo.Context) error {
		return c.String(http.StatusOK, "pong")
	})

	targets := app.http.Group("/macros")
	targets.Use(
		auth.NewAuthMiddleware(
			app.cfg.Authentication.BypassAuthentication,
			app.cfg.Authentication.AllowedTVMClients,
			app.cfg.Authentication.AllowedLogins,
		),
	)
	targets.GET("/:id", app.macrosEndpoint)
}

func (app App) Start() {
	// start cache sync
	app.netmapCache.StartCron()
	app.macrosCache.StartCron()
	app.cvsCache.StartCron()
	app.ypCache.StartCron()

	// start http app
	app.http.Logger.Fatal(app.http.Start(fmt.Sprintf(":%d", app.cfg.HTTPPort)))
}
