package auth

import (
	"context"
	"fmt"
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	// "a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
)

func sessionIDAuth(c echo.Context, bb httpbb.Client, acl map[string]bool) error {
	sessID, err := c.Cookie("Session_id")
	if err != nil {
		return xerrors.Errorf("failed to get Session_id cookie: %w", err)
	}

	bbResponse, err := bb.SessionID(c.Request().Context(), blackbox.SessionIDRequest{
		SessionID: sessID.Value,
		UserIP:    c.RealIP(),
		Host:      c.Request().Host,
	})
	if err != nil {
		return xerrors.Errorf("failed to check bbResponse session: %w", err)
	}

	if _, ok := acl[bbResponse.User.Login]; !ok {
		return xerrors.Errorf("this user have no access to this service")
	}

	return nil
}

func oauthAuth(c echo.Context, bb httpbb.Client, acl map[string]bool) error {
	token := c.Request().Header.Get("Authorization")
	if len(token) <= 6 || token[:6] != "OAuth " {
		return xerrors.New("failed to get OAuth token from Authorization header")
	}

	bbResponse, err := bb.OAuth(c.Request().Context(), blackbox.OAuthRequest{
		OAuthToken: token[6:],
		UserIP:     c.RealIP(),
	})
	if err != nil {
		return xerrors.Errorf("failed to check oauth token: %w", err)
	}

	if _, ok := acl[bbResponse.User.Login]; !ok {
		return xerrors.Errorf("this user have no access to this service")
	}

	return nil
}

func tvmAuth(c echo.Context, tvmClient tvmtool.Client, acl map[tvm.ClientID]bool) error {
	tvmTicket := c.Request().Header.Get("X-Ya-Service-Ticket")
	if len(tvmTicket) < 1 {
		return xerrors.Errorf("failed to get service ticket header")
	}

	serviceTicketStruct, err := tvmClient.CheckServiceTicket(context.Background(), tvmTicket)
	if err != nil {
		return xerrors.Errorf("service ticket is invalid: %w", err)
	}

	if _, ok := acl[serviceTicketStruct.SrcID]; !ok {
		return xerrors.Errorf("tvm client id is not allowed: %d", serviceTicketStruct.SrcID)
	}

	return nil
}

func NewAuthMiddleware(
	bypassAuthentication bool,
	allowedTVMClients []tvm.ClientID,
	allowedLogins []string,
) func(next echo.HandlerFunc) echo.HandlerFunc {

	allowedTVMClientsMap := make(map[tvm.ClientID]bool)
	for _, tvmClientID := range allowedTVMClients {
		allowedTVMClientsMap[tvmClientID] = true
	}

	allowedLoginsMap := make(map[string]bool)
	for _, login := range allowedLogins {
		allowedLoginsMap[login] = true
	}

	if bypassAuthentication {
		return func(next echo.HandlerFunc) echo.HandlerFunc {
			return func(c echo.Context) error {
				return next(c)
			}
		}
	} else {
		logger := func() log.Logger {
			zlog, err := zap.New(zap.ConsoleConfig(log.InfoLevel))
			if err != nil {
				panic(fmt.Sprintf("failed to create logger: %s", err))
			}
			return zlog
		}()

		tvmClient := func() *tvmtool.Client {
			tvmClient, err := tvmtool.NewDeployClient(tvmtool.WithLogger(logger.Structured()))
			if err != nil {
				panic(err)
			}
			return tvmClient
		}()

		bb := func() *httpbb.Client {
			bb, err := httpbb.NewIntranet(
				httpbb.WithLogger(logger.Structured()),
				httpbb.WithTVM(tvmClient),
			)
			if err != nil {
				panic(err)
			}
			return bb
		}()

		return func(next echo.HandlerFunc) echo.HandlerFunc {
			return func(c echo.Context) error {
				if sessionIDAuth(c, *bb, allowedLoginsMap) != nil &&
					oauthAuth(c, *bb, allowedLoginsMap) != nil &&
					tvmAuth(c, *tvmClient, allowedTVMClientsMap) != nil {
					return echo.NewHTTPError(http.StatusUnauthorized, "Please provide valid credentials")
				} else {
					return next(c)
				}
			}
		}
	}
}
