package cvs

import (
	"sync"

	"github.com/jasonlvhit/gocron"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/debby/targets/internal/models"
)

type CvsCache struct {
	cron      *gocron.Scheduler
	cvsClient CvsClient
	logger    log.Logger
	mu        sync.Mutex
	targets   []models.Target
	l3IPv6    []models.Target
}

type CacheOption func(nc *CvsCache)

func NewCvsCache(cvsClient CvsClient, options ...CacheOption) *CvsCache {
	cron := gocron.NewScheduler()

	cvsCache := CvsCache{
		cron:      cron,
		cvsClient: cvsClient,
		mu:        sync.Mutex{},
		targets:   []models.Target{},
		l3IPv6:    []models.Target{},
	}

	for _, op := range options {
		op(&cvsCache)
	}

	return &cvsCache
}

func CacheWithLogger(logger log.Logger) CacheOption {
	return func(cc *CvsCache) {
		cc.logger = logger
	}
}

func (cc *CvsCache) UpdateCache() {
	cc.logger.Fmt().Infof("CvsCache. updateCache started...")
	defer cc.logger.Fmt().Infof("CvsCache. updateCache finished...")

	newTargets, err := cc.cvsClient.FetchTargets()
	if err != nil {
		cc.logger.Fmt().Infof("error on target fetching: %v", err)
		return
	}

	cc.mu.Lock()
	cc.targets = newTargets
	cc.mu.Unlock()

	l3IPv6, err := cc.cvsClient.FetchL3IPs()
	if err != nil {
		cc.logger.Fmt().Infof("error on l3ipv6 fetching: %v", err)
		return
	}

	cc.mu.Lock()
	cc.l3IPv6 = l3IPv6
	cc.mu.Unlock()
}

func (cc *CvsCache) StartCron() {
	cc.logger.Fmt().Infof("cron job started...")
	cc.cron.Every(1).Hour().Do(cc.UpdateCache)
	go func() {
		<-cc.cron.Start()
	}()
}

// func (cc *CvsCache) StopCron() {
// 	cc.logger.Fmt().Infof("cron job stopped...")
// 	cc.cron.Clear()
// }

func (cc *CvsCache) GetTargets() []models.Target {
	cc.mu.Lock()
	targets := make([]models.Target, len(cc.targets))
	copy(targets, cc.targets)
	cc.mu.Unlock()
	return targets
}

func (cc *CvsCache) GetL3IPv6Targets() []models.Target {
	cc.mu.Lock()
	l3IPv6 := make([]models.Target, len(cc.l3IPv6))
	copy(l3IPv6, cc.l3IPv6)
	cc.mu.Unlock()
	return l3IPv6
}

func (cc *CvsCache) IsReady() bool {
	cc.mu.Lock()
	isReady := len(cc.targets) > 0 && len(cc.l3IPv6) > 0
	cc.mu.Unlock()
	return isReady
}
