package cvs

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"math/rand"
	"net"
	"os"
	"os/exec"
	"path"
	"regexp"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/debby/targets/internal/models"
)

type CvsClient struct {
	logger   log.Logger
	keyPath  string
	username string
}

type Option func(nc *CvsClient)

const letterBytes = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"

func randStringBytes(n int) string {
	b := make([]byte, n)
	for i := range b {
		b[i] = letterBytes[rand.Intn(len(letterBytes))]
	}
	return string(b)
}

func NewCvsClient(options ...Option) CvsClient {
	logger := &nop.Logger{}

	cvsClient := CvsClient{
		logger: logger,
	}

	for _, op := range options {
		op(&cvsClient)
	}

	return cvsClient
}

func WithLogger(logger log.Logger) Option {
	return func(cvsClient *CvsClient) {
		cvsClient.logger = logger
	}
}

func WithUsername(username string) Option {
	return func(cvsClient *CvsClient) {
		cvsClient.username = username
	}
}

func WithKeyPath(keyPath string) Option {
	return func(cvsClient *CvsClient) {
		cvsClient.keyPath = keyPath
	}
}

func (c CvsClient) parseDNSCache(dat string) []models.Target {
	lines := strings.Split(dat, "\n")
	targets := []models.Target{}
	for _, line := range lines {
		if len(line) == 0 {
			continue
		}
		parts := strings.Split(line, " ")
		fqdn := parts[0]
		ips := strings.Split(parts[1], ",")
		for _, ipStr := range ips {
			ip := net.ParseIP(ipStr)
			if ip == nil {
				c.logger.Fmt().Infof("error in parsing dnscache in line: '%s'", line)
			} else {
				targets = append(targets, models.Target{IP: ip, FQDN: &fqdn})
			}
		}
	}
	return targets
}

func (c CvsClient) FetchTargets() ([]models.Target, error) {
	// run ssh-agent
	cmd := exec.Command("ssh-agent")
	var out bytes.Buffer
	cmd.Stdout = &out
	err := cmd.Run()
	if err != nil {
		return nil, err
	}

	// extract ssh-agent evnironments from output
	s := out.String()
	parts := strings.Split(s, ";\n")
	if len(parts) != 4 {
		return nil, fmt.Errorf("ssh-agent returned unexcepted output")
	}
	envParts := strings.Split(parts[0], ";")
	if len(envParts) != 2 {
		return nil, fmt.Errorf("ssh-agent returned unexcepted output")
	}
	sshAuthSock := envParts[0]
	envParts = strings.Split(parts[1], ";")
	if len(envParts) != 2 {
		return nil, fmt.Errorf("ssh-agent returned unexcepted output")
	}
	sshAgentPid := envParts[0]

	env := append(os.Environ(), sshAuthSock, sshAgentPid)

	// defer ssh-agent kill
	defer func() {
		cmd := exec.Command("ssh-agent", "-k")
		cmd.Env = env
		_ = cmd.Run()
	}()

	// add ssh key to ssh-agent
	cmd = exec.Command("ssh-add", c.keyPath)
	cmd.Env = env
	if err := cmd.Run(); err != nil {
		return nil, err
	}

	// prepare folder and filename path
	foldername := randStringBytes(32)
	filename := path.Join(foldername, "router.dnscache.full")

	// checkout dnscache file
	cmd = exec.Command("cvs", "-d", fmt.Sprintf(":ext:%s@tree.yandex.ru:/opt/CVSROOT", c.username), "co", "-d", foldername, "noc/routers/fw/router.dnscache.full")
	cmd.Env = env
	if err := cmd.Run(); err != nil {
		return nil, err
	}

	// read dnscache file
	dat, err := ioutil.ReadFile(filename)
	if err != nil {
		return nil, err
	}

	// cleanup folder
	if err := os.RemoveAll(foldername); err != nil {
		return nil, err
	}

	// parse dnscache file
	targets := c.parseDNSCache(string(dat))
	return targets, nil
}

func (c CvsClient) FetchL3IPs() ([]models.Target, error) {
	// run ssh-agent
	cmd := exec.Command("ssh-agent")
	var out bytes.Buffer
	cmd.Stdout = &out
	err := cmd.Run()
	if err != nil {
		return nil, err
	}

	// extract ssh-agent evnironments from output
	s := out.String()
	parts := strings.Split(s, ";\n")
	if len(parts) != 4 {
		return nil, fmt.Errorf("ssh-agent returned unexcepted output")
	}
	envParts := strings.Split(parts[0], ";")
	if len(envParts) != 2 {
		return nil, fmt.Errorf("ssh-agent returned unexcepted output")
	}
	sshAuthSock := envParts[0]
	envParts = strings.Split(parts[1], ";")
	if len(envParts) != 2 {
		return nil, fmt.Errorf("ssh-agent returned unexcepted output")
	}
	sshAgentPid := envParts[0]

	env := append(os.Environ(), sshAuthSock, sshAgentPid)

	// defer ssh-agent kill
	defer func() {
		cmd := exec.Command("ssh-agent", "-k")
		cmd.Env = env
		_ = cmd.Run()
	}()

	// add ssh key to ssh-agent
	cmd = exec.Command("ssh-add", c.keyPath)
	cmd.Env = env
	if err := cmd.Run(); err != nil {
		return nil, err
	}

	// prepare folder and filename path
	foldername := randStringBytes(32)
	filename := path.Join(foldername, "services.yaml")

	// checkout dnscache file
	cmd = exec.Command("cvs", "-d", fmt.Sprintf(":ext:%s@tree.yandex.ru:/opt/CVSROOT", c.username), "co", "-d", foldername, "noc/balancers/iptables/services.yaml")
	cmd.Env = env
	if err := cmd.Run(); err != nil {
		return nil, err
	}

	// read services.yaml file
	dat, err := ioutil.ReadFile(filename)
	if err != nil {
		return nil, err
	}

	// cleanup folder
	if err := os.RemoveAll(foldername); err != nil {
		return nil, err
	}

	// parse ipv6 with regex
	return c.parseAllIPv6(string(dat))
}

func (c CvsClient) parseAllIPv6(dat string) ([]models.Target, error) {

	regexIPv6, err := regexp.Compile(`([\da-fA-F:]+)*:([\da-fA-F:]+)*:([\da-fA-F:]+)*`)
	if err != nil {
		return nil, err
	}

	targets := []models.Target{}
	findings := regexIPv6.FindAllString(dat, -1)
	for _, finding := range findings {
		ip := net.ParseIP(finding)
		if ip != nil {
			targets = append(targets, models.Target{IP: ip})
		}
	}

	return targets, nil
}
