package l3mgr

import (
	"sync"

	"github.com/jasonlvhit/gocron"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/debby/targets/internal/models"
)

type L3mgrCache struct {
	cron        *gocron.Scheduler
	l3mgrClient *L3mgrClient
	logger      log.Logger
	mu          sync.Mutex
	vsIPs       []models.Target
}

type CacheOption func(nc *L3mgrCache)

func WithCacheLogger(logger log.Logger) CacheOption {
	return func(c *L3mgrCache) {
		c.logger = logger
	}
}

func NewCache(l3mgrClient *L3mgrClient, options ...CacheOption) *L3mgrCache {
	cron := gocron.NewScheduler()
	logger := &nop.Logger{}

	l3mgrCache := L3mgrCache{
		cron:        cron,
		l3mgrClient: l3mgrClient,
		logger:      logger,
		mu:          sync.Mutex{},
		vsIPs:       []models.Target{},
	}

	for _, op := range options {
		op(&l3mgrCache)
	}

	return &l3mgrCache
}

func (c *L3mgrCache) UpdateCache() {
	c.logger.Fmt().Infof("L3mgrCache. updateCache started...")
	defer c.logger.Fmt().Infof("L3mgrCache. updateCache finished...")

	l3mgrServices, err := c.l3mgrClient.FetchServices()
	if err != nil {
		c.logger.Fmt().Infof("error on fetching l3mgrServices: %v", err)
		return
	}

	ipMap := make(map[string]interface{})
	for _, l3mgrService := range l3mgrServices {
		serviceDetails, err := c.l3mgrClient.FetchServiceDetails(l3mgrService.ID)
		if err != nil {
			c.logger.Fmt().Infof("error on fetching l3mgrService %d details: %v", l3mgrService.ID, err)
			return
		}
		for _, vs := range serviceDetails.VS {
			ipMap[vs.IP] = struct{}{}
		}
	}

	newVSIPs := []models.Target{}
	for ipStr := range ipMap {
		target := models.Parse(ipStr)
		if target.IP == nil {
			continue
		}

		ipv4 := target.IP.To4()
		if ipv4 != nil {
			continue
		}

		newVSIPs = append(newVSIPs, target)
	}

	c.mu.Lock()
	c.vsIPs = newVSIPs
	c.mu.Unlock()
}

func (c *L3mgrCache) StartCron() {
	c.logger.Fmt().Infof("cron job started...")
	c.cron.Every(12).Hours().Do(c.UpdateCache)
	go func() {
		<-c.cron.Start()
	}()
}

func (c *L3mgrCache) IsReady() bool {
	c.mu.Lock()
	isReady := len(c.vsIPs) > 0
	c.mu.Unlock()
	return isReady
}

func (c *L3mgrCache) GetVSIPs() []models.Target {
	c.mu.Lock()
	vsIPs := make([]models.Target, len(c.vsIPs))
	copy(vsIPs, c.vsIPs)
	c.mu.Unlock()
	return vsIPs
}
