package l3mgr

import (
	"crypto/tls"
	"fmt"
	"strconv"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

type L3mgrVS struct {
	ID int    `json:"id"`
	IP string `json:"ip"`
}

type L3mgrService struct {
	ID      int       `json:"id"`
	FDQN    string    `json:"fqdn"`
	ABC     string    `json:"abc"`
	State   string    `json:"state"`
	Archive bool      `json:"archive"`
	URL     string    `json:"url"`
	VS      []L3mgrVS `json:"vs"`
}

type APIResponseServices struct {
	Objects []L3mgrService `json:"objects"`
	Page    int            `json:"page"`
	Limit   int            `json:"limit"`
	Total   int            `json:"total"`
}

type L3mgrClient struct {
	httpc          *resty.Client
	logger         log.Logger
	pageLimit      int
	pagesHardLimit int
}

type Option func(nc *L3mgrClient)

func WithLogger(logger log.Logger) Option {
	return func(c *L3mgrClient) {
		c.logger = logger
		c.httpc.SetLogger(logger.Fmt())
	}
}

func NewClient(oauthToken string, options ...Option) (*L3mgrClient, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, err
	}

	logger := &nop.Logger{}
	httpClient := resty.New().
		SetRetryCount(2).
		SetLogger(logger.Fmt()).
		SetHeader("Authorization", "OAuth "+oauthToken).
		SetHeader("User-Agent", "debby-targets <security@yandex-team.ru>").
		SetTLSClientConfig(&tls.Config{RootCAs: certPool})

	l3mgrClient := L3mgrClient{
		httpc:          httpClient,
		logger:         logger,
		pageLimit:      1000,
		pagesHardLimit: 500,
	}

	for _, op := range options {
		op(&l3mgrClient)
	}

	return &l3mgrClient, nil
}

func (c L3mgrClient) FetchServices() ([]L3mgrService, error) {
	l3mgrServices := []L3mgrService{}

	var page int
	for page = 1; page <= c.pagesHardLimit; page++ {
		var resp APIResponseServices

		r, err := c.httpc.R().
			SetQueryParams(map[string]string{
				"archive": strconv.FormatBool(false),
				"_full":   strconv.FormatBool(true),
				"_limit":  strconv.Itoa(c.pageLimit),
				"_page":   strconv.Itoa(page),
			}).
			SetResult(&resp).
			Get("https://l3-api.tt.yandex-team.ru/api/v1/service")

		if err != nil {
			return nil, err
		}

		if !r.IsSuccess() {
			return nil, fmt.Errorf("error on fetching l3mgr services. status code: %d", r.StatusCode())
		}

		l3mgrServices = append(l3mgrServices, resp.Objects...)

		if len(resp.Objects) < c.pageLimit {
			break
		}
	}

	if page == c.pagesHardLimit {
		c.logger.Fmt().Warnf("warn of fetching l3mgr services: pageHardLimit (=%d) exceeded.", c.pagesHardLimit)
	}

	return l3mgrServices, nil
}

func (c L3mgrClient) FetchServiceDetails(serviceID int) (*L3mgrService, error) {
	var resp L3mgrService

	r, err := c.httpc.R().
		SetPathParams(map[string]string{
			"serviceID": strconv.Itoa(serviceID),
		}).
		SetResult(&resp).
		Get("https://l3-api.tt.yandex-team.ru/api/v1/service/{serviceID}")

	if err != nil {
		return nil, err
	}

	if !r.IsSuccess() {
		return nil, fmt.Errorf("error on fetching l3mgr service %d. status code: %d", serviceID, r.StatusCode())
	}

	return &resp, nil
}
