package macros

import (
	"sync"

	"github.com/jasonlvhit/gocron"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/debby/targets/internal/models"
)

type MacrosCache struct {
	cron            *gocron.Scheduler
	logger          log.Logger
	macrosClient    *MacrosClient
	macrosToTargets map[string][]models.Target
	mu              sync.Mutex
}

type CacheOption func(nc *MacrosCache)

func NewMacrosCache(macrosClient *MacrosClient, options ...CacheOption) *MacrosCache {
	cron := gocron.NewScheduler()

	macrosCache := MacrosCache{
		cron:            cron,
		macrosClient:    macrosClient,
		macrosToTargets: make(map[string][]models.Target),
		mu:              sync.Mutex{},
	}

	for _, op := range options {
		op(&macrosCache)
	}

	return &macrosCache
}

func CacheWithLogger(logger log.Logger) CacheOption {
	return func(mc *MacrosCache) {
		mc.logger = logger
	}
}

func (mc *MacrosCache) UpdateCache() {
	mc.logger.Fmt().Infof("MacrosCache. updateCache started...")
	defer mc.logger.Fmt().Infof("MacrosCache. updateCache finished...")

	newMacrosToTargets, err := mc.macrosClient.FetchMacros()
	if err != nil {
		mc.logger.Fmt().Infof("error on macros fetching: %v", err)
		return
	}

	mc.mu.Lock()
	mc.macrosToTargets = newMacrosToTargets
	mc.mu.Unlock()
}

func (mc *MacrosCache) StartCron() {
	mc.logger.Fmt().Infof("cron job started...")
	mc.cron.Every(1).Hour().Do(mc.UpdateCache)
	go func() {
		<-mc.cron.Start()
	}()
}

// func (mc *MacrosCache) StopCron() {
// 	mc.logger.Fmt().Infof("cron job stopped...")
// 	mc.cron.Clear()
// }

func (mc *MacrosCache) GetTargets(macro string) ([]models.Target, bool) {

	mc.mu.Lock()
	targets, ok := mc.macrosToTargets[macro]
	mc.mu.Unlock()

	if !ok {
		mc.logger.Fmt().Infof("macro %s not found", macro)
		return nil, false
	}

	return targets, true
}

func (mc *MacrosCache) ExistsMacro(macro string) bool {

	mc.mu.Lock()
	_, ok := mc.macrosToTargets[macro]
	mc.mu.Unlock()
	return ok
}

func (mc *MacrosCache) IsReady() bool {
	mc.mu.Lock()
	isReady := len(mc.macrosToTargets) > 0
	mc.mu.Unlock()
	return isReady
}
