package macros

import (
	"crypto/tls"
	// "encoding/json"
	"fmt"
	"net"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/debby/targets/internal/models"
)

type MacrosClient struct {
	httpc  *resty.Client
	logger log.Logger
}

type Option func(nc *MacrosClient)

func NewMacrosClient(options ...Option) (*MacrosClient, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, err
	}

	logger := &nop.Logger{}
	httpClient := resty.New().
		SetRetryCount(2).
		SetLogger(logger.Fmt()).
		SetHeader("User-Agent", "debby-targets <security@yandex-team.ru>").
		SetTLSClientConfig(&tls.Config{RootCAs: certPool})

	macrosClient := MacrosClient{
		httpc:  httpClient,
		logger: logger,
	}

	for _, op := range options {
		op(&macrosClient)
	}

	return &macrosClient, nil
}

func WithLogger(logger log.Logger) Option {
	return func(macrosClient *MacrosClient) {
		macrosClient.logger = logger
		macrosClient.httpc.SetLogger(logger.Fmt())
	}
}

func (mc MacrosClient) FetchMacros() (map[string][]models.Target, error) {
	results := make(map[string][]string)

	resp, err := mc.httpc.R().
		SetQueryParams(map[string]string{
			"net_only": "0",
		}).
		SetResult(&results).
		Get("https://hbf.yandex.net/dump-all-macros")

	if err != nil {
		mc.logger.Fmt().Infof("error on fetching: %s", err)
		return nil, err
	}

	if !resp.IsSuccess() {
		mc.logger.Fmt().Infof("error on fetching: %v", resp.IsSuccess())
		return nil, fmt.Errorf("error on fetching")
	}

	macrosToTargets := make(map[string][]models.Target)

	for macro, targets := range results {
		macrosToTargets[macro] = []models.Target{}
		for _, targetStr := range targets {
			target := models.Target{}
			ip := net.ParseIP(targetStr)
			if ip != nil {
				target.IP = ip
			} else {
				parts := strings.Split(targetStr, "@")
				var projectID *string
				var netStr *string
				if len(parts) == 1 {
					netStr = &parts[0]
				} else {
					projectID = &parts[0]
					netStr = &parts[1]
				}
				_, net, err := net.ParseCIDR(*netStr)
				if err != nil {
					// mc.logger.Fmt().Infof("error %v parsing cidr %s.", err, *netStr)
					target.FQDN = &targetStr
				} else {
					target.ProjectID = projectID
					target.NET = net
				}
			}
			macrosToTargets[macro] = append(macrosToTargets[macro], target)
		}
	}

	return macrosToTargets, nil
}
