package netmap

import (
	"sync"

	"github.com/jasonlvhit/gocron"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/debby/targets/internal/models"
)

type NetmapCache struct {
	cron         *gocron.Scheduler
	logger       log.Logger
	mu           sync.Mutex
	netmapClient *NetmapClient
	targets      []models.Target
}

type CacheOption func(nc *NetmapCache)

func NewNetmapCache(netmapClient *NetmapClient, options ...CacheOption) *NetmapCache {
	cron := gocron.NewScheduler()

	netmapCache := NetmapCache{
		cron:         cron,
		netmapClient: netmapClient,
		targets:      []models.Target{},
		mu:           sync.Mutex{},
	}

	for _, op := range options {
		op(&netmapCache)
	}

	return &netmapCache
}

func CacheWithLogger(logger log.Logger) CacheOption {
	return func(nc *NetmapCache) {
		nc.logger = logger
	}
}

func (nc *NetmapCache) UpdateCache() {
	nc.logger.Fmt().Infof("NetmapCache. updateCache started...")
	defer nc.logger.Fmt().Infof("NetmapCache. updateCache finished...")

	newTargets, err := nc.netmapClient.FetchTargets()
	if err != nil {
		nc.logger.Fmt().Infof("error on target fetching: %v", err)
		return
	}

	nc.mu.Lock()
	nc.targets = newTargets
	nc.mu.Unlock()
}

func (nc *NetmapCache) StartCron() {
	nc.logger.Fmt().Infof("cron job started...")
	nc.cron.Every(1).Hour().Do(nc.UpdateCache)
	go func() {
		<-nc.cron.Start()
	}()
}

// func (nc *NetmapCache) StopCron() {
// 	nc.logger.Fmt().Infof("cron job stopped...")
// 	nc.cron.Clear()
// }

func (nc *NetmapCache) GetTargets() []models.Target {
	nc.mu.Lock()
	targets := make([]models.Target, len(nc.targets))
	copy(targets, nc.targets)
	nc.mu.Unlock()
	return targets
}

func (nc *NetmapCache) IsReady() bool {
	nc.mu.Lock()
	isReady := len(nc.targets) > 0
	nc.mu.Unlock()
	return isReady
}
