package netmap

import (
	"crypto/tls"
	"fmt"
	"net"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/debby/targets/internal/models"
)

type NetmapClient struct {
	httpc  *resty.Client
	logger log.Logger
}

type Option func(nc *NetmapClient)

func NewNetmapClient(options ...Option) (*NetmapClient, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, err
	}

	logger := &nop.Logger{}
	httpClient := resty.New().
		SetRetryCount(2).
		SetLogger(logger.Fmt()).
		SetHeader("User-Agent", "debby-targets <security@yandex-team.ru>").
		SetTLSClientConfig(&tls.Config{RootCAs: certPool})

	netmapClient := NetmapClient{
		httpc:  httpClient,
		logger: logger,
	}

	for _, op := range options {
		op(&netmapClient)
	}

	return &netmapClient, nil
}

func WithLogger(logger log.Logger) Option {
	return func(netmapClient *NetmapClient) {
		netmapClient.logger = logger
		netmapClient.httpc.SetLogger(logger.Fmt())
	}
}

func (nc NetmapClient) parseNetmap(dat string) []models.Target {
	lines := strings.Split(dat, "\n")

	targets := []models.Target{}
	for _, line := range lines {
		if len(line) == 0 {
			continue
		}

		lineParts := strings.Split(line, "\t")
		if len(lineParts) != 2 {
			nc.logger.Fmt().Infof("error in line parsing: %s", line)
		} else {
			ipStr := lineParts[0]
			ip := net.ParseIP(ipStr)
			if ip == nil {
				nc.logger.Fmt().Infof("error in parsing ip: %s", ipStr)
			} else {
				targets = append(targets, models.Target{IP: ip})
			}
		}
	}
	return targets
}

func (nc NetmapClient) FetchTargets() ([]models.Target, error) {
	resp, err := nc.httpc.R().Get("https://ro.racktables.yandex.net/export/netmap/L23.v46")

	if err != nil {
		nc.logger.Fmt().Infof("error on fetching: %s", err)
		return nil, err
	}

	if !resp.IsSuccess() {
		nc.logger.Fmt().Infof("error on fetching. status: %s", resp.Status())
		return nil, fmt.Errorf("error on fetching. status: %s", resp.Status())
	}

	body := string(resp.Body())
	targets := nc.parseNetmap(body)
	return targets, nil
}
