package utils

import (
	"fmt"
	"net"
	"strings"

	"a.yandex-team.ru/security/debby/targets/internal/models"
)

func extractProjectID(ip net.IP) string {
	b1 := int(ip[8])
	b2 := int(ip[9])
	b3 := int(ip[10])
	b4 := int(ip[11])
	projectID := fmt.Sprintf("%02x%02x%02x%02x", b1, b2, b3, b4)
	return strings.TrimLeft(projectID, "0")
}

func IsIPv6(ip net.IP) bool {
	s := ip.String()
	for i := 0; i < len(s); i++ {
		switch s[i] {
		case '.':
			return false
		case ':':
			return true
		}
	}
	return false
}

func RemainOnlyKnownIPV6Targets(knownTargets, allTargets []models.Target) []models.Target {
	remainedTargets := []models.Target{}

	for _, target := range allTargets {
		if target.NET != nil && IsIPv6(target.NET.IP) {
			for _, knownTarget := range knownTargets {
				if knownTarget.IP != nil && IsIPv6(knownTarget.IP) {
					if target.NET.Contains(knownTarget.IP) {
						if target.ProjectID == nil {
							remainedTargets = append(remainedTargets, knownTarget)
						} else if extractProjectID(knownTarget.IP) == *target.ProjectID {
							remainedTargets = append(remainedTargets, knownTarget)
						}
					}
				}
			}
		} else {
			remainedTargets = append(remainedTargets, target)
		}
	}

	return remainedTargets
}

func DumpsTargets(targets []models.Target) string {
	var dumpedTargets []string
	for _, target := range targets {
		dumpedTargets = append(dumpedTargets, target.Dumps())
	}
	return strings.Join(dumpedTargets, ",")
}

func IsMacroAcceptable(macro string, acceptable []string) bool {
	for _, acceptableMacro := range acceptable {
		if strings.Compare(macro, acceptableMacro) == 0 {
			return true
		}
	}
	return false
}

func isYandexnets(target models.Target, yandexnets6Targets []models.Target) bool {
	for _, yn6Target := range yandexnets6Targets {
		switch {
		case target.IP == nil: // skip not ip targets
			return true
		case target.IP.To4() != nil: // skip not ipv6 targets
			return true
		case yn6Target.NET != nil && yn6Target.NET.Contains(target.IP): // allow ips with are in yandexnets6 networks
			return true
		case yn6Target.IP != nil && yn6Target.IP.Equal(target.IP): // allow ips with are in yandexnets6 hosts
			return true
		}
	}
	return false
}

func RemainUniqueTargets(targets []models.Target, yandexnets6Targets []models.Target) []models.Target {
	if targets == nil {
		return nil
	}
	uniqueMap := make(map[string]models.Target)
	for _, t := range targets {
		if !isYandexnets(t, yandexnets6Targets) {
			continue
		}
		s := t.Dumps()
		if _, ok := uniqueMap[s]; !ok {
			uniqueMap[s] = t
		}
	}
	unique := []models.Target{}
	for _, t := range uniqueMap {
		unique = append(unique, t)
	}

	return unique
}

func RemainUniqueTargetsNoShuffle(targets []models.Target, yandexnets6Targets []models.Target) []models.Target {
	if targets == nil {
		return nil
	}
	unique := []models.Target{}
	uniqueMap := make(map[string]models.Target)
	for _, t := range targets {
		if !isYandexnets(t, yandexnets6Targets) {
			continue
		}
		s := t.Dumps()
		if _, ok := uniqueMap[s]; !ok {
			uniqueMap[s] = t
			unique = append(unique, t)
		}
	}

	return unique
}
