package yp

import (
	"sync"

	"github.com/jasonlvhit/gocron"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/debby/targets/internal/models"
)

type YpCache struct {
	cli                            YpClient
	cron                           *gocron.Scheduler
	logger                         log.Logger
	mu                             sync.Mutex
	clusterToEndpointIP6Addresses  map[string][]models.Target
	clusterToIP6AddressAllocations map[string]IP6AddressAllocations
}

type CacheOption func(nc *YpCache)

func NewYpCache(cli YpClient, options ...CacheOption) *YpCache {
	cron := gocron.NewScheduler()

	cache := YpCache{
		cron:                           cron,
		cli:                            cli,
		mu:                             sync.Mutex{},
		clusterToEndpointIP6Addresses:  map[string][]models.Target{},
		clusterToIP6AddressAllocations: map[string]IP6AddressAllocations{},
	}

	for _, op := range options {
		op(&cache)
	}

	return &cache
}

func CacheWithLogger(logger log.Logger) CacheOption {
	return func(cc *YpCache) {
		cc.logger = logger
	}
}

func (cc *YpCache) UpdateCache() {
	cc.logger.Fmt().Infof("YpCache. updateCache started...")
	defer cc.logger.Fmt().Infof("YpCache. updateCache finished...")

	clusterToEndpointIP6Addresses := cc.cli.FetchEndpointIP6Addresses()
	cc.mu.Lock()
	cc.clusterToEndpointIP6Addresses = clusterToEndpointIP6Addresses
	cc.mu.Unlock()

	clusterToIP6AddressAllocations := cc.cli.FetchIP6AddressAllocations()
	cc.mu.Lock()
	cc.clusterToIP6AddressAllocations = clusterToIP6AddressAllocations
	cc.mu.Unlock()
}

func (cc *YpCache) StartCron() {
	cc.logger.Fmt().Infof("cron job started...")
	cc.cron.Every(1).Hour().Do(cc.UpdateCache)
	go func() {
		<-cc.cron.Start()
	}()
}

// func (cc *CvsCache) StopCron() {
// 	cc.logger.Fmt().Infof("cron job stopped...")
// 	cc.cron.Clear()
// }

func (cc *YpCache) GetTargets() []models.Target {
	cc.mu.Lock()
	results := []models.Target{}
	for _, v := range cc.clusterToEndpointIP6Addresses {
		results = append(results, v...)
	}
	for _, ip6AddressAllocations := range cc.clusterToIP6AddressAllocations {
		results = append(results, ip6AddressAllocations.VMBackbone...)
		results = append(results, ip6AddressAllocations.VMFastbone...)
		results = append(results, ip6AddressAllocations.ContainerBackbone...)
		results = append(results, ip6AddressAllocations.ContainerFastbone...)
		results = append(results, ip6AddressAllocations.Others...)
	}
	cc.mu.Unlock()
	return results
}

func (cc *YpCache) IsReady() bool {
	cc.mu.Lock()
	isReady := len(cc.clusterToEndpointIP6Addresses) == len(cc.cli.clusters)
	isReady = isReady && len(cc.clusterToIP6AddressAllocations) == len(cc.cli.clusters)
	cc.mu.Unlock()
	return isReady
}
