package yp

import (
	"context"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/debby/targets/internal/models"
	"a.yandex-team.ru/yp/go/yp"
	"a.yandex-team.ru/yp/go/yson/ypapi"
)

type IP6AddressAllocations struct {
	VMBackbone        []models.Target
	VMFastbone        []models.Target
	ContainerBackbone []models.Target
	ContainerFastbone []models.Target
	Others            []models.Target
}

type YpClient struct {
	clusters []string
	logger   log.Logger
}

type Option func(nc *YpClient)

func NewYpClient(options ...Option) YpClient {

	cli := YpClient{
		clusters: []string{"sas", "vla", "myt", "man", "iva"},
		logger:   &nop.Logger{},
	}

	for _, op := range options {
		op(&cli)
	}

	return cli
}

func WithLogger(logger log.Logger) Option {
	return func(c *YpClient) {
		c.logger = logger
	}
}

func (cli YpClient) FetchEndpointIP6AddressesForCluster(cluster string) ([]models.Target, error) {

	ypClient, err := yp.NewClient(
		cluster,
		yp.WithSystemAuthToken(),
		yp.WithLogger(cli.logger),
	)
	if err != nil {
		return nil, err
	}
	defer ypClient.Close()

	resp, err := ypClient.SelectObjects(
		context.Background(),
		yp.SelectObjectsRequest{
			ObjectType: yp.ObjectTypeEndpoint,
			Format:     yp.PayloadFormatYson,
			Selectors:  []string{"/spec/ip6_address"},
		},
	)
	if err != nil {
		return nil, err
	}

	results := []models.Target{}
	for resp.Next() {
		var address string
		if err := resp.Fill(&address); err != nil {
			return nil, err
		}
		target := models.Parse(address)
		if target.IP != nil {
			results = append(results, target)
		}
	}

	return results, nil
}

func (cli YpClient) FetchEndpointIP6Addresses() map[string][]models.Target {
	clusterToEndpointIP6Addresses := map[string][]models.Target{}

	for _, cluster := range cli.clusters {
		targets, err := cli.FetchEndpointIP6AddressesForCluster(cluster)
		if err != nil {
			cli.logger.Fmt().Infof("Error fetching endpoint's ip6_address for cluster %s: %#v", cluster, err)
		} else {
			clusterToEndpointIP6Addresses[cluster] = targets
		}
	}

	return clusterToEndpointIP6Addresses
}

func (cli YpClient) FetchIP6AddressAllocationsForCluster(cluster string) (*IP6AddressAllocations, error) {

	ypClient, err := yp.NewClient(
		cluster,
		yp.WithSystemAuthToken(),
		yp.WithLogger(cli.logger),
	)
	if err != nil {
		return nil, err
	}
	defer ypClient.Close()

	continuationToken := ""

	results := map[string][]models.Target{
		"vm_backbone":        []models.Target{},
		"vm_fastbone":        []models.Target{},
		"container_backbone": []models.Target{},
		"container_fastbone": []models.Target{},
		"others":             []models.Target{},
	}

	for {
		req := yp.SelectObjectsRequest{
			ObjectType: yp.ObjectTypePod,
			Format:     yp.PayloadFormatYson,
			Selectors:  []string{"/status/ip6_address_allocations"},
			Limit:      1000,
		}
		if len(continuationToken) > 0 {
			req.ContinuationToken = continuationToken
		}

		resp, err := ypClient.SelectObjects(context.Background(), req)
		if err != nil {
			return nil, err
		}

		for resp.Next() {
			var ipv6Allocations []ypapi.TPodStatus_TIP6AddressAllocation
			if err := resp.Fill(&ipv6Allocations); err != nil {
				panic(err)
			}
			for _, allocation := range ipv6Allocations {

				if allocation.Address == nil {
					continue
				}

				target := models.Parse(*allocation.Address)
				if target.IP == nil {
					continue
				}

				if allocation.VlanId == nil {
					results["others"] = append(results["others"], target)
					continue
				}

				switch {
				case allocation.Labels["owner"] == "vm" && *allocation.VlanId == "backbone":
					results["vm_backbone"] = append(results["vm_backbone"], target)
				case allocation.Labels["owner"] == "vm" && *allocation.VlanId == "fastbone":
					results["vm_fastbone"] = append(results["vm_fastbone"], target)
				case allocation.Labels["owner"] == "container" && *allocation.VlanId == "backbone":
					results["container_backbone"] = append(results["container_backbone"], target)
				case allocation.Labels["owner"] == "container" && *allocation.VlanId == "fastbone":
					results["container_fastbone"] = append(results["container_fastbone"], target)
				default:
					results["others"] = append(results["others"], target)
				}
			}
		}

		if resp.Count() < 1000 {
			break
		}

		continuationToken = resp.ContinuationToken()
	}

	ipAllocations := &IP6AddressAllocations{
		VMBackbone:        results["vm_backbone"],
		VMFastbone:        results["vm_fastbone"],
		ContainerBackbone: results["container_backbone"],
		ContainerFastbone: results["container_fastbone"],
		Others:            results["others"],
	}

	return ipAllocations, nil
}

func (cli YpClient) FetchIP6AddressAllocations() map[string]IP6AddressAllocations {
	clusterToIP6AddressAllocations := map[string]IP6AddressAllocations{}

	for _, cluster := range cli.clusters {
		ip6AddressAllocations, err := cli.FetchIP6AddressAllocationsForCluster(cluster)
		if err != nil {
			cli.logger.Fmt().Infof("Error fetching ip6_address_allocations for cluster %s: %#v", cluster, err)
		} else {
			clusterToIP6AddressAllocations[cluster] = *ip6AddressAllocations
		}
	}

	return clusterToIP6AddressAllocations
}
